package me.ahoo.eventbus.jdbc;

import com.google.common.base.Throwables;
import lombok.var;
import me.ahoo.eventbus.core.repository.ConcurrentVersionConflictException;
import me.ahoo.eventbus.core.publisher.PublishEventWrapper;
import me.ahoo.eventbus.core.repository.entity.SubscribeEventCompensationEntity;
import me.ahoo.eventbus.core.repository.RepeatedSubscribeException;
import me.ahoo.eventbus.core.subscriber.Subscriber;
import me.ahoo.eventbus.core.repository.*;
import me.ahoo.eventbus.core.repository.entity.SubscribeEventEntity;
import me.ahoo.eventbus.core.utils.Dates;
import me.ahoo.eventbus.core.utils.Jsons;
import org.springframework.dao.EmptyResultDataAccessException;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;
import org.springframework.jdbc.support.GeneratedKeyHolder;

import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.Optional;

/**
 * @author ahoo wang
 */
public class JdbcSubscribeEventRepository implements SubscribeEventRepository {
    private final NamedParameterJdbcTemplate jdbcTemplate;

    public JdbcSubscribeEventRepository(NamedParameterJdbcTemplate jdbcTemplate) {

        this.jdbcTemplate = jdbcTemplate;
    }

    private static final String SQL_GET_SUBSCRIBE_EVENT = "select id,status,version from subscribe_event " +
            "where event_id=:event_id and event_name=:event_name and subscribe_name=:subscribe_name limit 1;";

    private Optional<SubscribeIdentity> getSubscribeIdentity(Subscriber subscriber, Long eventId, String eventName) {
        MapSqlParameterSource getSubscribeEventParams = new MapSqlParameterSource("event_id", eventId);
        getSubscribeEventParams.addValue("event_name", eventName);
        getSubscribeEventParams.addValue("subscribe_name", subscriber.getName());
        try {
            SubscribeIdentity subscribeIdentity = jdbcTemplate.queryForObject(SQL_GET_SUBSCRIBE_EVENT, getSubscribeEventParams, (rs, rowNum) -> {
                var id = rs.getLong("id");
                var status = rs.getInt("status");
                var version = rs.getInt("version");
                return SubscribeIdentity.builder()
                        .id(id)
                        .subscriberName(subscriber.getName())
                        .status(SubscribeStatus.valeOf(status))
                        .version(version)
                        .build();
            });
            return Optional.of(subscribeIdentity);
        } catch (EmptyResultDataAccessException ex) {
            return Optional.empty();
        }
    }

    private static final String SQL_SUBSCRIBE_INITIALIZED = "insert subscribe_event(subscribe_name, status, event_id, event_name, event_data, event_create_time, version)" +
            "values (:subscribe_name, :status, :event_id, :event_name, :event_data, :event_create_time, :version);";

    private SubscribeIdentity initializeSubscribeIdentity(Subscriber subscriber, PublishEventWrapper subscribePublishEventWrapper, String eventName) {
        var subscribeIdentity = SubscribeIdentity.builder()
                .status(SubscribeStatus.INITIALIZED)
                .version(Version.INITIAL_VALUE)
                .subscriberName(subscriber.getName())
                .build();
        MapSqlParameterSource sqlParams = new MapSqlParameterSource("subscribe_name", subscriber.getName());
        sqlParams.addValue("status", subscribeIdentity.getStatus().getValue());
        sqlParams.addValue("event_id", subscribePublishEventWrapper.getId());
        sqlParams.addValue("event_name", eventName);
        String eventData = Jsons.serializeAsString(subscribePublishEventWrapper.getEventData());
        sqlParams.addValue("event_data", eventData);
        sqlParams.addValue("event_create_time", subscribePublishEventWrapper.getCreateTime());
        sqlParams.addValue("version", subscribeIdentity.getVersion());
        var keyHolder = new GeneratedKeyHolder();
        jdbcTemplate.update(SQL_SUBSCRIBE_INITIALIZED, sqlParams, keyHolder);
        subscribeIdentity.setId(keyHolder.getKey().longValue());
        return subscribeIdentity;
    }

    @Override
    public SubscribeIdentity initialize(Subscriber subscriber, PublishEventWrapper subscribePublishEventWrapper) throws RepeatedSubscribeException {
        String eventName = subscribePublishEventWrapper.getEventName();
        var subscribeIdentity = getSubscribeIdentity(subscriber, subscribePublishEventWrapper.getId(), eventName);
        if (subscribeIdentity.isPresent()) {
            if (subscribeIdentity.get().getStatus().equals(SubscribeStatus.SUCCEEDED)) {
                throw new RepeatedSubscribeException(subscriber, subscribePublishEventWrapper);
            }
            return subscribeIdentity.get();
        }
        return initializeSubscribeIdentity(subscriber, subscribePublishEventWrapper, eventName);
    }

    private static final String SQL_MARK_STATUS
            = "update subscribe_event set status=:status,version=version+1 where id=:id and version=:version;";

    @Override
    public int markStatus(SubscribeIdentity subscribeIdentity) {
        MapSqlParameterSource sqlParams = new MapSqlParameterSource("id", subscribeIdentity.getId());
        sqlParams.addValue("status", subscribeIdentity.getStatus().getValue());
        sqlParams.addValue("version", subscribeIdentity.getVersion());
        var affected = jdbcTemplate.update(SQL_MARK_STATUS, sqlParams);
        if (affected == 0) {
            var errMsg = String.format("Subscribe [%s] mark [%d]@[%d] to status [%s] error."
                    , subscribeIdentity.getSubscriberName()
                    , subscribeIdentity.getId()
                    , subscribeIdentity.getVersion()
                    , subscribeIdentity.getStatus().name());
            throw new ConcurrentVersionConflictException(errMsg, subscribeIdentity);
        }
        return affected;
    }

    @Override
    public int markSucceeded(SubscribeIdentity subscribeIdentity) {
        subscribeIdentity.setStatus(SubscribeStatus.SUCCEEDED);
        return markStatus(subscribeIdentity);
    }

    private static final String SQL_SUBSCRIBE_FAILED
            = "insert subscribe_event_failed (subscribe_event_id, failed_msg) values (:subscribe_event_id, :failed_msg)";

    /**
     * 1. insert failed log
     * 2. mark failed status
     *
     * @param subscribeIdentity
     * @param throwable
     * @return
     */
    @Override
    public int markFailed(SubscribeIdentity subscribeIdentity, Throwable throwable) {

        MapSqlParameterSource sqlParams = new MapSqlParameterSource("subscribe_event_id", subscribeIdentity.getId());
        var failedMsg = Throwables.getStackTraceAsString(throwable);
        sqlParams.addValue("failed_msg", failedMsg);
        jdbcTemplate.update(SQL_SUBSCRIBE_FAILED, sqlParams);

        subscribeIdentity.setStatus(SubscribeStatus.FAILED);
        return markStatus(subscribeIdentity);
    }

    private static final String SQL_QUERY_FAILED
            = "select id, subscribe_name, status, subscribe_time, event_id, event_name, event_data, event_create_time, version, create_time from subscribe_event " +
            "where status<>1 and create_time<:before and version<:max_version order by version asc limit :limit;";

    @Override
    public List<SubscribeEventEntity> queryFailed(int limit, int before, int maxVersion) {
        MapSqlParameterSource sqlParams = new MapSqlParameterSource("max_version", maxVersion);
        var beforeDate = LocalDateTime.now().minus(before, ChronoUnit.MINUTES);
        sqlParams.addValue("before", beforeDate);
        sqlParams.addValue("limit", limit);
        return jdbcTemplate.query(SQL_QUERY_FAILED, sqlParams, (rs, rowNum) -> {
            var id = rs.getLong("id");
            var subscribeName = rs.getString("subscribe_name");
            var status = rs.getInt("status");
            var subscribeTime = rs.getTimestamp("subscribe_time");
            var eventId = rs.getLong("event_id");
            var eventName = rs.getString("event_name");
            var eventDataStr = rs.getString("event_data");
            var eventCreateTime = rs.getTimestamp("event_create_time");
            var version = rs.getInt("version");
            var createTime = rs.getTimestamp("create_time");

            var entity = new SubscribeEventEntity();
            entity.setId(id);
            entity.setSubscriberName(subscribeName);
            entity.setStatus(SubscribeStatus.valeOf(status));
            entity.setSubscribeTime(Dates.of(subscribeTime));
            entity.setEventId(eventId);
            entity.setEventName(eventName);
            entity.setEventData(eventDataStr);

            entity.setEventCreateTime(Dates.of(eventCreateTime));
            entity.setVersion(version);
            entity.setCreateTime(Dates.of(createTime));
            return entity;
        });
    }
    private static String SQL_COMPENSATE
            = "insert subscribe_event_compensation (subscribe_event_id, start_time, taken, failed_msg) " +
            "values (:subscribe_event_id, :start_time, :taken, :failed_msg) ";
    @Override
    public int compensate(SubscribeEventCompensationEntity subscribeEventCompensationEntity) {
        MapSqlParameterSource sqlParams = new MapSqlParameterSource("subscribe_event_id", subscribeEventCompensationEntity.getSubscribeEventId());
        sqlParams.addValue("start_time", subscribeEventCompensationEntity.getStartTime());
        sqlParams.addValue("taken", subscribeEventCompensationEntity.getTaken());
        sqlParams.addValue("failed_msg", subscribeEventCompensationEntity.getFailedMsg());
        return jdbcTemplate.update(SQL_COMPENSATE, sqlParams);
    }
}
