package me.ahoo.eventbus.rabbit;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import lombok.var;
import me.ahoo.eventbus.core.compensation.CompensationPublishEventWrapper;
import me.ahoo.eventbus.core.publisher.PublishEventWrapper;
import me.ahoo.eventbus.core.utils.Dates;
import me.ahoo.eventbus.core.utils.Jsons;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageBuilder;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.core.MessagePropertiesBuilder;

import java.util.Date;

public abstract class EventCodecs {

    public static final String EVENT_ID = "event_id";
    public static final String EVENT_NAME = "event_name";
    public static final String EVENT_CREATE_TIME = "event_create_time";

    public static Message encode(PublishEventWrapper publishEventWrapper) {
        var eventName = publishEventWrapper.getEventName();

        MessageProperties messageProperties = MessagePropertiesBuilder.newInstance()
                .setHeaderIfAbsent(EVENT_ID, publishEventWrapper.getId())
                .setHeaderIfAbsent(EVENT_NAME, eventName)
                .setHeaderIfAbsent(EVENT_CREATE_TIME, Dates.of(publishEventWrapper.getCreateTime())).build();

        byte[] eventBuff;
        if (publishEventWrapper instanceof CompensationPublishEventWrapper) {
            eventBuff = ((CompensationPublishEventWrapper) publishEventWrapper).getEventData().getBytes(Charsets.UTF_8);
        } else {
            eventBuff = Jsons.serialize(publishEventWrapper.getEventData());
        }
        return MessageBuilder.withBody(eventBuff).andProperties(messageProperties).build();
    }

    public static PublishEventWrapper decode(Message message, Class<?> eventClass) {
        var messageProperties = message.getMessageProperties();
        Long eventId = messageProperties.getHeader(EVENT_ID);
        Preconditions.checkNotNull(eventId, "%s can not be null.", EVENT_ID);

        String eventName = messageProperties.getHeader(EVENT_NAME);
        Preconditions.checkNotNull(eventName, "%s can not be null.", EVENT_NAME);

        Date eventCreateTimeDate = messageProperties.getHeader(EVENT_CREATE_TIME);
        Preconditions.checkNotNull(eventCreateTimeDate, "%s can not be null.", EVENT_CREATE_TIME);

        var eventCreateTime = Dates.of(eventCreateTimeDate);

        var typedEventData = Jsons.deserialize(message.getBody(), eventClass);

        return new PublishEventWrapper(eventId, eventName, typedEventData, eventCreateTime);

    }
}
