package me.ahoo.eventbus.rabbit;

import com.google.common.base.Charsets;
import com.google.common.base.Preconditions;
import lombok.var;
import me.ahoo.eventbus.core.compensation.CompensationPublishEvent;
import me.ahoo.eventbus.core.publisher.PublishEvent;
import me.ahoo.eventbus.core.utils.Dates;
import me.ahoo.eventbus.core.utils.Jsons;
import org.springframework.amqp.core.Message;
import org.springframework.amqp.core.MessageBuilder;
import org.springframework.amqp.core.MessageProperties;
import org.springframework.amqp.core.MessagePropertiesBuilder;

import java.util.Date;

public abstract class EventCodecs {

    public static final String EVENT_ID = "event_id";
    public static final String EVENT_NAME = "event_name";
    public static final String EVENT_CREATE_TIME = "event_create_time";

    public static Message encode(PublishEvent publishEvent) {
        var eventName = publishEvent.getEventName();

        MessageProperties messageProperties = MessagePropertiesBuilder.newInstance()
                .setHeaderIfAbsent(EVENT_ID, publishEvent.getId())
                .setHeaderIfAbsent(EVENT_NAME, eventName)
                .setHeaderIfAbsent(EVENT_CREATE_TIME, Dates.of(publishEvent.getCreateTime())).build();

        byte[] eventBuff;
        if (publishEvent instanceof CompensationPublishEvent) {
            eventBuff = ((CompensationPublishEvent) publishEvent).getEventData().getBytes(Charsets.UTF_8);
        } else {
            eventBuff = Jsons.serialize(publishEvent.getEventData());
        }
        return MessageBuilder.withBody(eventBuff).andProperties(messageProperties).build();
    }

    public static PublishEvent decode(Message message, Class<?> eventDataClass) {
        var messageProperties = message.getMessageProperties();
        Long eventId = messageProperties.getHeader(EVENT_ID);
        Preconditions.checkNotNull(eventId, "%s can not be null.", EVENT_ID);

        String eventName = messageProperties.getHeader(EVENT_NAME);
        Preconditions.checkNotNull(eventName, "%s can not be null.", EVENT_NAME);

        Date eventCreateTimeDate = messageProperties.getHeader(EVENT_CREATE_TIME);
        Preconditions.checkNotNull(eventCreateTimeDate, "%s can not be null.", EVENT_CREATE_TIME);

        var eventCreateTime = Dates.of(eventCreateTimeDate);

        var typedEventData = Jsons.deserialize(message.getBody(), eventDataClass);

        return new PublishEvent(eventId, eventName, typedEventData, eventCreateTime);

    }
}
