package me.ahoo.pigeon.connector.starter.core;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import lombok.var;
import me.ahoo.cosky.discovery.InstanceIdGenerator;
import me.ahoo.cosky.discovery.spring.cloud.registry.CoskyRegistration;
import me.ahoo.pigeon.connector.core.config.Connector;
import me.ahoo.pigeon.connector.discovery.ConnectorDiscovery;
import me.ahoo.pigeon.connector.starter.netty.NettyProperties;
import me.ahoo.pigeon.core.connector.ConnectorId;
import me.ahoo.pigeon.core.expression.SpringExpression;

import javax.annotation.PostConstruct;
import java.util.Objects;

/**
 * @author ahoo wang
 */
@Slf4j
public class ConnectorRegistrationProcessor {

    private final Connector connector;
    private final NettyProperties nettyProperties;
    private final ConnectorId connectorId;
    private final CoskyRegistration registration;

    public ConnectorRegistrationProcessor(Connector connector,
                                          NettyProperties nettyProperties,
                                          ConnectorId connectorId,
                                          CoskyRegistration registration) {
        this.connector = connector;
        this.nettyProperties = nettyProperties;
        this.connectorId = connectorId;
        this.registration = registration;
    }

    @PostConstruct
    public void process() {

        var instance = registration.of();
        if (instance.getPort() == 0) {
            instance.setPort(connector.getPort());
        }

        if (Objects.nonNull(nettyProperties.getSsl())) {
            instance.setSchema("wss");
        } else {
            instance.setSchema("ws");
        }

        processExpose();
        ConnectorDiscovery.setConnectorId(registration.getMetadata(), this.connectorId.getId());

        instance.setInstanceId(InstanceIdGenerator.DEFAULT.generate(instance));
    }

    private void processExpose() {
        var expose = connector.getExpose();
        if (Objects.isNull(expose)) {
            if (log.isWarnEnabled()) {
                log.warn("pigeon.connector.expose is null,skip setting connector.expose.");
                return;
            }
        }
        var connectorMetadata = this.registration.getMetadata();
        if (!Strings.isNullOrEmpty(expose.getSchema())) {
            ConnectorDiscovery.setConnectorExposeSchema(connectorMetadata, expose.getSchema());
        }

        if (!Objects.isNull(expose.getPort())) {
            ConnectorDiscovery.setConnectorExposePort(connectorMetadata, expose.getPort());
        }

        String host = expose.getHost();
        if (Strings.isNullOrEmpty(host) && !Strings.isNullOrEmpty(expose.getHostTemplate())) {
            var stringExpression = new SpringExpression(expose.getHostTemplate());
            host = stringExpression.parse(connectorId);
        }
        if (!Strings.isNullOrEmpty(host)) {
            ConnectorDiscovery.setConnectorExposeHost(registration.getMetadata(), host);
        }

    }
}
