package me.ahoo.pigeon.core.bus.impl;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import lombok.var;
import me.ahoo.pigeon.core.bus.subscriber.SubscriberDefinition;
import me.ahoo.pigeon.core.message.Message;
import me.ahoo.pigeon.core.message.RouteDirection;
import me.ahoo.pigeon.core.Constants;
import me.ahoo.pigeon.core.bus.TopicParser;

import java.util.Objects;
import java.util.Set;

/**
 * @author ahoo wang
 * Creation time 2020/10/19 23:49
 **/
public class DefaultTopicParser implements TopicParser {

    /**
     * default : {@link Constants#PIGEON_PREFIX}
     */
    private final String topicPrefix;
    /**
     * {@link #topicPrefix}{@link RouteDirection#CONNECTOR}
     */
    private final String connectorTopic;
    /**
     * {@link #connectorTopic}.{connectorId}
     */
    private final String connectorTopicFormat;
    /**
     * {@link #topicPrefix}}{@link RouteDirection#ROUTER}
     */
    private final String routerTopic;

    public DefaultTopicParser() {
        this(Constants.PIGEON_PREFIX);
    }

    public DefaultTopicParser(String topicPrefix) {
        this.topicPrefix = topicPrefix;
        this.routerTopic = topicPrefix + RouteDirection.ROUTER.name().toLowerCase();
        this.connectorTopic = topicPrefix + RouteDirection.CONNECTOR.name().toLowerCase();
        this.connectorTopicFormat = connectorTopic + ".%s";
    }

    /**
     * Parse message topic
     *
     * @param message
     * @return topic
     */
    @Override
    public String parseMessageTopic(Message message) {
        final var direction = message.getDirection(true);
        switch (direction) {
            case CONNECTOR: {
                var connectorId = message.getHeader().getReceiverConnectorId(false);
                if (Objects.nonNull(connectorId)) {
                    return Strings.lenientFormat(connectorTopicFormat, connectorId);
                }
                return connectorTopic;
            }
            case COMMANDER: {
                return getCommanderTopic(message.getCommandType(true));
            }
            case ROUTER: {
                return routerTopic;
            }
            default:
                throw new IllegalStateException("Unexpected value: " + direction);
        }
    }

    /**
     * Parse topics of {@link SubscriberDefinition}
     *
     * @param subscriberDefinition
     * @return
     */
    @Override
    public Set<String> parseSubscriberTopic(SubscriberDefinition subscriberDefinition) {
        final var direction = subscriberDefinition.getDirection();
        switch (direction) {
            case CONNECTOR: {
                final var topicOfConnectorId = Strings.lenientFormat(connectorTopicFormat, subscriberDefinition.getConnectorId());
                return Sets.newHashSet(connectorTopic, topicOfConnectorId);
            }
            case COMMANDER: {
                final var commandType = subscriberDefinition.getCommandType();
                return Sets.newHashSet(getCommanderTopic(commandType));
            }
            case ROUTER: {
                return Sets.newHashSet(routerTopic);
            }
            default:
                throw new IllegalStateException("Unexpected value: " + direction);
        }
    }

    protected String getCommanderTopic(String commandType) {
        return topicPrefix + commandType;
    }
}
