package me.ahoo.pigeon.core.bus.impl;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import lombok.var;
import me.ahoo.pigeon.core.bus.Dispatcher;
import me.ahoo.pigeon.core.bus.MessageBus;
import me.ahoo.pigeon.core.bus.MessageTopicParser;
import me.ahoo.pigeon.core.bus.PublishResult;
import me.ahoo.pigeon.core.bus.subscriber.Subscriber;
import me.ahoo.pigeon.core.bus.subscriber.SubscriberRegistry;
import me.ahoo.pigeon.core.message.Message;
import me.ahoo.pigeon.core.util.Futures;

import java.util.concurrent.CompletableFuture;

/**
 * @author ahoo wang
 * Creation time: 2019/11/22 13:17
 */
@Slf4j
public class MemoryMessageBus implements MessageBus {

    private final Dispatcher dispatcher;
    private final MessageTopicParser messageTopicParser;
    private final SubscriberRegistry subscriberRegistry;

    public MemoryMessageBus(Dispatcher dispatcher,
                            MessageTopicParser messageTopicParser,
                            SubscriberRegistry subscriberRegistry) {
        this.dispatcher = dispatcher;
        this.messageTopicParser = messageTopicParser;
        this.subscriberRegistry = subscriberRegistry;
    }

    @Override
    public CompletableFuture<PublishResult> publish(Message message) {
        try {
            var topic = messageTopicParser.parseMessageTopic(message);
            var subscribers = subscriberRegistry.getSubscribers(topic);
            dispatcher.dispatch(message, subscribers);
            if (log.isDebugEnabled()) {
                log.debug("publish - message.id:[{}] ,direction:[{}],commandType:[{}]",
                        message.getHeader().getId(false), message.getDirection(false), message.getCommandType(false));
            }
            return CompletableFuture.completedFuture(new PublishResult(message));
        } catch (Throwable throwable) {
            log.error(Strings.lenientFormat("publish - failed! -> id:[%s] ,direction:[%s],commandType:[%s]",
                    message.getHeader().getId(false), message.getDirection(false), message.getCommandType(false)), throwable);

            return Futures.ofFailed(throwable);
        }
    }

    @Override
    public void subscribe(Subscriber subscriber) {
        subscriberRegistry.register(subscriber);
    }

}
