package me.ahoo.pigeon.core.bus.subscriber.impl;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import me.ahoo.pigeon.core.message.Message;
import me.ahoo.pigeon.core.message.RouteDirection;
import me.ahoo.pigeon.core.bus.subscriber.ExceptionHandler;
import me.ahoo.pigeon.core.bus.subscriber.Subscriber;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Objects;
import java.util.Set;

/**
 * @author ahoo wang
 * Creation time: 2020/3/20 10:33
 */
@Slf4j
public class DefaultSubscriber implements Subscriber {
    private final String name;
    private final Object target;
    private final Method method;
    private final RouteDirection direction;
    private final String commandType;
    private final String groupId;
    private final Set<String> topics;
    private final Integer connectorId;
    private final ExceptionHandler exceptionHandler;

    public DefaultSubscriber(String name, Object target, Method method, RouteDirection direction, String commandType, String groupId, Set<String> topics, Integer connectorId, ExceptionHandler exceptionHandler) {
        Preconditions.checkNotNull(name, "name can not be null!");
        Preconditions.checkNotNull(target, "target can not be null!");
        Preconditions.checkNotNull(method, "method can not be null!");
        Preconditions.checkState(Objects.nonNull(direction) && !RouteDirection.NULL.equals(direction), "direction can not be null!");
        if (RouteDirection.COMMANDER.equals(direction)) {
            Preconditions.checkState(!Strings.isNullOrEmpty(commandType), "the commandType can not be null or empty  when the direction is {@link RouteDirection#COMMANDER}!");
        }
        Preconditions.checkState(Objects.nonNull(topics) && !topics.isEmpty(), "topics can not be null or empty!");
        Preconditions.checkState(!Strings.isNullOrEmpty(groupId), "groupId can not be null or empty!");
        if (RouteDirection.CONNECTOR.equals(direction)) {
            Preconditions.checkState(Objects.nonNull(connectorId), "the connectorId can not be null when the direction is {@link RouteDirection#CONNECTOR}!");
        }
        this.name = name;
        this.target = target;
        this.method = method;
        if (!this.method.isAccessible()) {
            this.method.setAccessible(true);
        }
        this.direction = direction;
        this.commandType = commandType;
        this.topics = topics;
        this.groupId = groupId;
        this.connectorId = connectorId;
        this.exceptionHandler = MoreObjects.firstNonNull(exceptionHandler, ExceptionHandler.LOGGER);
    }

    @Override
    public void invoke(Message message) {
        try {
            if (log.isDebugEnabled())   {
                log.debug("invoke -> [{}] .", message);
            }
            method.invoke(target, message);
        } catch (InvocationTargetException invocationTargetEx) {
            exceptionHandler.handle(this, message, invocationTargetEx.getTargetException());
        } catch (Exception ex) {
            exceptionHandler.handle(this, message, ex);
        }
    }

    @Override
    public String getName() {
        return this.name;
    }

    @Override
    public Object getTarget() {
        return this.target;
    }

    @Override
    public Method getMethod() {
        return this.method;
    }

    @Override
    public RouteDirection getDirection() {
        return this.direction;
    }

    @Override
    public String getCommandType() {
        return this.commandType;
    }

    @Override
    public String getGroupId() {
        return this.groupId;
    }

    @Override
    public Set<String> getTopics() {
        return this.topics;
    }

    @Override
    public Integer getConnectorId() {
        return this.connectorId;
    }

    @Override
    public ExceptionHandler getExceptionHandler() {
        return this.exceptionHandler;
    }

    @Override
    public String toString() {
        return name;
    }
}
