package me.ahoo.pigeon.core.bus.subscriber.impl;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.var;
import me.ahoo.pigeon.core.bus.subscriber.*;
import me.ahoo.pigeon.core.connector.ConnectorId;
import me.ahoo.pigeon.core.message.Message;
import me.ahoo.pigeon.core.message.RouteDirection;
import me.ahoo.pigeon.core.bus.Subscribe;
import org.springframework.aop.support.AopUtils;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * @author ahoo wang
 * Creation time: 2020/3/20 10:30
 */
public class DefaultSubscriberFactory implements SubscriberFactory {
    private static final int ONE = 1;

    private final ConnectorId connectorId;
    private final SubscriberGroupIdParser subscriberGroupIdParser;
    private final SubscriberTopicParser subscriberTopicParser;
    private final ExceptionHandler exceptionHandler;

    public DefaultSubscriberFactory(ConnectorId connectorId, SubscriberGroupIdParser subscriberGroupIdParser, SubscriberTopicParser subscriberTopicParser, ExceptionHandler exceptionHandler) {
        this.connectorId = connectorId;
        this.subscriberGroupIdParser = subscriberGroupIdParser;
        this.subscriberTopicParser = subscriberTopicParser;
        this.exceptionHandler = exceptionHandler;
    }

    @Override
    public List<Subscriber> getSubscribers(Object targetObj) {
        return parse(targetObj).stream().map(this::getSubscriber).collect(Collectors.toList());
    }


    private List<SubscriberDefinition> parse(Object targetObj) {
        Preconditions.checkNotNull(targetObj);

        var targetClass = AopUtils.getTargetClass(targetObj);

        var classSubscribe = AnnotationUtils.findAnnotation(targetClass, Subscribe.class);
        List<SubscriberDefinition> subscriberDefinitions = new ArrayList<>();
        ReflectionUtils.doWithMethods(targetClass, method -> {

            var methodSubscribe = AnnotationUtils.getAnnotation(method, Subscribe.class);
            if (Objects.isNull(methodSubscribe)) {
                return;
            }
            Preconditions.checkState(method.getParameterCount() == ONE, "method:[%s] ParameterCount must be 1.", method);
            var firstParameterType = method.getParameterTypes()[0];
            Preconditions.checkState(Message.class.isAssignableFrom(firstParameterType), "method:[%s] parameter.Type must be Message.");

            var subscriberDefinition = parseMethod(targetObj, method, methodSubscribe, classSubscribe);
            subscriberDefinitions.add(subscriberDefinition);
        }, ReflectionUtils.USER_DECLARED_METHODS);
        return subscriberDefinitions;
    }

    private SubscriberDefinition parseMethod(Object targetObject, Method method, Subscribe methodSubscribe, Subscribe classSubscribe) {

        var definitionBuilder = SubscriberDefinition.builder().target(targetObject).method(method).exceptionHandler(exceptionHandler);

        var name = methodSubscribe.name();
        if (Objects.nonNull(classSubscribe) && !Strings.isNullOrEmpty(classSubscribe.name())) {
            name = classSubscribe.name() + methodSubscribe.name();
        }
        if (Strings.isNullOrEmpty(name)) {
            name = generateSubscriberName(method);
        }
        definitionBuilder.name(name);

        var direction = methodSubscribe.direction();
        if (RouteDirection.NULL.equals(direction)) {
            Preconditions.checkNotNull(classSubscribe);
            Preconditions.checkState(!RouteDirection.NULL.equals(classSubscribe.direction()));
            direction = classSubscribe.direction();
        }
        definitionBuilder.direction(direction);
        if (RouteDirection.CONNECTOR.equals(direction)) {
            definitionBuilder.connectorId(connectorId.getId());
        }

        var commandType = methodSubscribe.commandType();
        if (Objects.nonNull(classSubscribe) && !Strings.isNullOrEmpty(classSubscribe.commandType())) {
            commandType = classSubscribe.commandType() + methodSubscribe.commandType();
        }
        definitionBuilder.commandType(commandType);
        var subscriberDefinition = definitionBuilder.build();

        var groupId = methodSubscribe.groupId();

        if (Strings.isNullOrEmpty(groupId)) {
            groupId = subscriberGroupIdParser.parseGroupId(subscriberDefinition);
        }

        if (Objects.nonNull(classSubscribe) && !Strings.isNullOrEmpty(classSubscribe.groupId())) {
            groupId = classSubscribe.groupId() + groupId;
        }
        subscriberDefinition.setGroupId(groupId);

        var topics = subscriberTopicParser.parseSubscriberTopic(subscriberDefinition);
        subscriberDefinition.setTopics(topics);

        return subscriberDefinition;
    }

    private String generateSubscriberName(Method subscribeMethod) {
        return subscribeMethod.getName() + "@" + subscribeMethod.getDeclaringClass().getName();
    }


    public Subscriber getSubscriber(SubscriberDefinition subscriberDefinition) {
        return subscriberDefinition.asSubscriber();
    }
}
