package me.ahoo.pigeon.core.codec;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import lombok.var;
import me.ahoo.pigeon.core.codec.decoder.CustomizeBodyMessageDecoder;
import me.ahoo.pigeon.core.codec.decoder.NoneMessageDecoder;
import me.ahoo.pigeon.core.codec.decoder.StringBodyMessageDecoder;
import me.ahoo.pigeon.core.message.CommandTypes;
import me.ahoo.pigeon.core.bus.subscriber.Subscriber;
import me.ahoo.pigeon.core.util.Messages;

import java.util.concurrent.ConcurrentHashMap;

/**
 * Decoder Registrar
 *
 * @author ahoo wang
 * Creation time: 2020/2/25 16:50
 */
@Slf4j
public class DecoderRegistrar {

    private ConcurrentHashMap<String, StringToMessageDecoder> typeDecoders = new ConcurrentHashMap<>();

    public DecoderRegistrar() {
        register(CommandTypes.CLIENT_ACK, NoneMessageDecoder.INSTANCE);
        register(CommandTypes.SERVER_ACK, NoneMessageDecoder.INSTANCE);
        register(CommandTypes.HEARTBEAT, NoneMessageDecoder.INSTANCE);
        register(CommandTypes.AUTH, NoneMessageDecoder.INSTANCE);
    }

    public void register(String commandType, StringToMessageDecoder messageDecoder) {

        var beforeVal = typeDecoders.put(commandType, messageDecoder);
        if (log.isInfoEnabled()) {
            log.info("register - commandType:[{}] , messageDecoder:[{}] , before:[{}]", commandType, messageDecoder.getClass().getName(), beforeVal);
        }
    }

    public void register(Subscriber subscriber) {
        String commandType = subscriber.getCommandType();
        if (Strings.isNullOrEmpty(commandType)) {
            if (log.isInfoEnabled()) {
                log.info("register - Ignore :commandType is null or empty, subscriber:[{}]", subscriber);
            }
            return;
        }

        typeDecoders.computeIfAbsent(commandType, (cmdType) -> {
            var bodyClass = Messages.getBodyClass(subscriber.getMethod());
            if (String.class.equals(bodyClass)) {
                if (log.isInfoEnabled()) {
                    log.info("register - commandType:[{}] , bodyClass:[{}], messageDecoder:[StringBodyMessageDecoder.INSTANCE].", commandType, bodyClass.getName());
                }
                return StringBodyMessageDecoder.INSTANCE;
            }
            if (Void.class.equals(bodyClass)) {
                if (log.isInfoEnabled()) {
                    log.info("register - commandType:[{}] , bodyClass:[{}], messageDecoder:[NoneMessageDecoder.INSTANCE].", commandType, bodyClass.getName());
                }
                return NoneMessageDecoder.INSTANCE;
            }
            if (log.isInfoEnabled()) {
                log.info("register - commandType:[{}] , bodyClass:[{}] with CustomizeBodyMessageDecoder.", commandType, bodyClass.getName());
            }
            return new CustomizeBodyMessageDecoder(bodyClass);
        });
    }

    public boolean contains(String commandType) {
        return typeDecoders.containsKey(commandType);
    }

    public StringToMessageDecoder find(String commandType) {
        return typeDecoders.getOrDefault(commandType, StringBodyMessageDecoder.INSTANCE);
    }
}
