package me.ahoo.pigeon.core.codec;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.SneakyThrows;
import lombok.var;
import me.ahoo.pigeon.core.message.Message;
import me.ahoo.pigeon.core.message.MessageHeader;
import me.ahoo.pigeon.core.util.Clazzs;
import me.ahoo.pigeon.core.util.ObjectMappers;

import java.util.Objects;

/**
 * @author ahoo wang
 * Creation time: 2019/11/21 11:23
 */
public abstract class StringToMessageDecoder<TBody> implements MessageDecoder<TBody, String, String, String> {
    protected final ObjectMapper objectMapper;
    private final Class<TBody> bodyClass;
    private final boolean noneBody;

    public StringToMessageDecoder() {
        this(ObjectMappers.of(), null);
    }

    public StringToMessageDecoder(ObjectMapper objectMapper, Class<TBody> bodyClass) {
        this.objectMapper = objectMapper;
        if (Objects.isNull(bodyClass)) {
            bodyClass = Clazzs.getFirstClassGenericType(this.getClass());
        }
        this.bodyClass = bodyClass;
        this.noneBody = bodyClass.equals(Void.class);
    }

    @Override
    public Message<TBody> decode(String source) {
        Preconditions.checkNotNull(source);
        var sourceMessageBag = new StringMessageBag(source);

        var msgObj = new Message<TBody>();
        var header = decodeHeader(sourceMessageBag.getHeader());
        msgObj.setHeader(header);
        if (noneBody) {
            return msgObj;
        }
        var body = decodeBody(header, sourceMessageBag.getBody());
        if (Objects.nonNull(body)) {
            msgObj.setBody(body);
        }
        return msgObj;
    }

    @SneakyThrows
    @Override
    public MessageHeader decodeHeader(String headerStr) {
        return objectMapper.readValue(headerStr, MessageHeader.class);
    }

    @SneakyThrows
    @Override
    public TBody decodeBody(MessageHeader header, String bodyStr) {
        if (Strings.isNullOrEmpty(bodyStr)) {
            return null;
        }
        if (bodyClass.isInstance(bodyStr)) {
            return (TBody) bodyStr;
        }
        return objectMapper.readValue(bodyStr, bodyClass);
    }
}
