package me.ahoo.pigeon.core.message;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.Getter;
import lombok.SneakyThrows;
import lombok.var;

import java.util.Objects;

/**
 * @author ahoo wang
 * Creation time: 2019/11/18 18:33
 */
public final class Message<TBody> implements Cloneable {
    /**
     * @see MessageHeader
     */
    @Getter
    private MessageHeader header;
    @Getter
    private TBody body;

    public Message<TBody> setHeader(MessageHeader header) {
        this.header = header;
        return this;
    }

    public Message<TBody> setBody(TBody body) {
        this.body = body;
        return this;
    }

    public RouteDirection getDirection(boolean required) {
        checkHeader();
        return header.getDirection(required);
    }

    private void checkHeader() {
        Preconditions.checkNotNull(header, "header can not be null.");
    }

    public Message<TBody> setDirection(RouteDirection routeDirection) {
        checkHeader();
        header.setDirection(routeDirection);
        return this;
    }

    public String getCommandType(boolean required) {
        checkHeader();
        return header.getCommandType(required);
    }

    public Message<TBody> setCommandType(String commandType) {
        checkHeader();
        header.setCommandType(commandType);
        return this;
    }

    public Long getId(boolean required) {
        checkHeader();
        return header.getId(required);
    }

    public Message<TBody> setId(Long id) {
        checkHeader();
        header.setId(id);
        return this;
    }

    public SendMode getSendMode(boolean required) {
        checkHeader();
        return header.getSendMode(required);
    }

    public Message<TBody> setSendMode(SendMode sendMode) {
        checkHeader();
        header.setSendMode(sendMode);
        return this;
    }

    //region App

    public String getApp(boolean required) {
        checkHeader();
        return header.getApp(required);
    }

    public Message<TBody> setApp(String app) {
        checkHeader();
        header.setApp(app);
        return this;
    }

    //endregion

    //region Error

    public String getErrorCode(boolean required) {
        checkHeader();
        return header.getErrorCode(required);
    }

    public Message<TBody> setErrorCode(String errorCode) {
        checkHeader();
        header.setErrorCode(errorCode);
        return this;
    }

    public String getErrorMsg(boolean required) {
        checkHeader();
        return header.getErrorMsg(required);
    }

    public Message<TBody> setErrorMsg(String errorCode) {
        checkHeader();
        header.setErrorMsg(errorCode);
        return this;
    }

    //endregion

    //region FilterExpression

    public String getFilterExpression(boolean required) {
        checkHeader();
        return header.getFilterExpression(required);
    }

    public Message<TBody> setFilterExpression(String filterExpression) {
        checkHeader();
        header.setFilterExpression(filterExpression);
        return this;
    }

    //endregion

    //region ClientMessageId

    public String getClientMessageId(boolean required) {
        checkHeader();
        return header.getClientMessageId(required);
    }

    public Message<TBody> setClientMessageId(String clientMessageId) {
        checkHeader();
        header.setClientMessageId(clientMessageId);
        return this;
    }

    //endregion

    //region ServerMessageId

    public Long getServerMessageId(boolean required) {
        checkHeader();
        return header.getServerMessageId(required);
    }

    public Message<TBody> setServerMessageId(Long serverMessageId) {
        checkHeader();
        header.setServerMessageId(serverMessageId);
        return this;
    }

    //endregion

    //region Retry

    public boolean isRetry() {
        checkHeader();
        return header.isRetry();
    }

    public Integer getRetryTimes(boolean required) {
        checkHeader();
        return header.getRetryTimes(required);
    }

    public Message<TBody> setRetryTimes(Integer retryTimes) {
        checkHeader();
        header.setRetryTimes(retryTimes);
        return this;
    }
    //endregion

    //region Sender

    public String getSenderChannelId(boolean required) {
        checkHeader();
        return header.getSenderChannelId(required);
    }

    public Message<TBody> setSenderChannelId(String senderChannelId) {
        checkHeader();
        header.setSenderChannelId(senderChannelId);
        return this;
    }

    public Long getSenderDeviceId(boolean required) {
        checkHeader();
        return header.getSenderDeviceId(required);
    }

    public Message<TBody> setSenderDeviceId(Long senderDeviceId) {
        checkHeader();
        header.setSenderDeviceId(senderDeviceId);
        return this;
    }

    public Long getSenderUserId(boolean required) {
        checkHeader();
        return header.getSenderUserId(required);
    }

    public Message<TBody> setSenderUserId(Long senderUserId) {
        checkHeader();
        header.setSenderUserId(senderUserId);
        return this;
    }

    public Integer getSenderConnectorId(boolean required) {
        checkHeader();
        return header.getSenderConnectorId(required);
    }

    public Message<TBody> setSenderConnectorId(Integer senderConnectorId) {
        checkHeader();
        header.setSenderConnectorId(senderConnectorId);
        return this;
    }

    public Long getCreateTime(boolean required) {
        checkHeader();
        return header.getCreateTime(required);
    }

    public Message<TBody> setCreateTime(Long createTime) {
        checkHeader();
        header.setCreateTime(createTime);
        return this;
    }

    /**
     * set {@link System#currentTimeMillis()} to create time
     *
     * @return
     */
    public Message<TBody> ofNow() {
        checkHeader();
        header.ofNow();
        return this;
    }
    //endregion

    //region Receiver

    public String getReceiverChannelId(boolean required) {
        checkHeader();
        return header.getReceiverChannelId(required);
    }

    public Message<TBody> setReceiverChannelId(String receiverChannelId) {
        checkHeader();
        header.setReceiverChannelId(receiverChannelId);
        return this;
    }

    public Long getReceiverDeviceId(boolean required) {
        checkHeader();
        return header.getReceiverDeviceId(required);
    }

    public Message<TBody> setReceiverDeviceId(Long receiverDeviceId) {
        checkHeader();
        header.setReceiverDeviceId(receiverDeviceId);
        return this;
    }

    public Long getReceiverUserId(boolean required) {
        checkHeader();
        return header.getReceiverUserId(required);
    }

    public Message<TBody> setReceiverUserId(Long receiverUserId) {
        checkHeader();
        header.setReceiverUserId(receiverUserId);
        return this;
    }

    public Long getReceiverGroupId(boolean required) {
        checkHeader();
        return header.getReceiverGroupId(required);
    }

    public Message<TBody> setReceiverGroupId(Long receiverGroupId) {
        checkHeader();
        header.setReceiverGroupId(receiverGroupId);
        return this;
    }

    public Long getReceiverRoomId(boolean required) {
        checkHeader();
        return header.getReceiverRoomId(required);
    }

    public Message<TBody> setReceiverRoomId(Long receiverRoomId) {
        checkHeader();
        header.setReceiverRoomId(receiverRoomId);
        return this;
    }

    public Integer getReceiverConnectorId(boolean required) {
        checkHeader();
        return header.getReceiverConnectorId(required);
    }

    public Message<TBody> setReceiverConnectorId(Integer receiverConnectorId) {
        checkHeader();
        header.setReceiverConnectorId(receiverConnectorId);
        return this;
    }

    //endregion

    //region static-new


    public static <TBody> Message<TBody> ofRouter() {
        return ofDirection(RouteDirection.ROUTER);
    }

    public static <TBody> Message<TBody> ofConnector() {
        return ofDirection(RouteDirection.CONNECTOR);
    }

    public static <TBody> Message<TBody> ofCommander() {
        return ofDirection(RouteDirection.COMMANDER);
    }

    public static <TBody> Message<TBody> ofDirection(RouteDirection routeDirection) {
        return new Message<TBody>().setHeader(new MessageHeader().setDirection(routeDirection));
    }

    public static <TBody> Message<TBody> ofCommandType(String commandType) {
        return new Message<TBody>().setHeader(new MessageHeader().setCommandType(commandType));
    }

    public static Message<Void> error(String errorCode, String errorMsg) {
        return error(errorCode, errorMsg, null);
    }

    public static Message<Void> error(String errorCode, String errorMsg, Message sourceMessage) {

        var errorMessage = Message.<Void>ofCommandType(CommandTypes.ERROR);
        errorMessage.setErrorCode(errorCode);
        if (Objects.nonNull(errorMsg)) {
            errorMessage.setErrorMsg(errorMsg);
        }
        if (Objects.nonNull(sourceMessage)) {
            var sourceMsgHeader = sourceMessage.getHeader();
            var sourceClientMsgId = sourceMsgHeader.getClientMessageId(false);
            if (!Strings.isNullOrEmpty(sourceClientMsgId)) {
                errorMessage.setClientMessageId(sourceClientMsgId);
            }
            var sourceSenderChannelId = sourceMsgHeader.getSenderChannelId(false);
            if (!Strings.isNullOrEmpty(sourceSenderChannelId)) {
                errorMessage.setReceiverChannelId(sourceSenderChannelId);
            }
            var sourceSenderConnectorId = sourceMsgHeader.getSenderConnectorId(false);
            if (Objects.nonNull(sourceSenderConnectorId)) {
                errorMessage.setReceiverConnectorId(sourceSenderConnectorId);
            }
        }

        return errorMessage;
    }

    //endregion

    /**
     * Clone Root and header
     *
     * @return new Message
     */
    @SneakyThrows
    @Override
    public Message<TBody> clone() {
        var msgClone = (Message<TBody>) super.clone();
        msgClone.header = (MessageHeader) header.clone();
        return msgClone;
    }

    @Override
    public String toString() {
        return "Message{" +
                "header=" + header +
                ", body=" + body +
                '}';
    }
}
