package me.ahoo.pigeon.core.message;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.primitives.Longs;
import lombok.var;

import java.util.HashMap;
import java.util.Objects;

/**
 * @author ahoo wang
 * Creation time: 2019/11/21 10:30
 */
public class MessageHeader extends HashMap<String, Object> {

    public <TValue> TValue getGenericValue(String key, Class<TValue> valClass, boolean required, Function<Object, TValue> converter) {
        var valObj = get(key);
        if (Objects.isNull(valObj)) {
            if (required) {
                throw new NullPointerException(String.format("Header:[%s] can not be null!", key));
            }
            return null;
        }
        if (valClass.isInstance(valObj)) {
            return (TValue) valObj;
        }
        var convertedVal = converter.apply(valObj);
        computeIfPresent(key, (k, v) -> convertedVal);
        return convertedVal;
    }

    public String getStringValue(String key, boolean required) {
        return getGenericValue(key, String.class, required, (obj) -> obj.toString());
    }

    public Integer getIntegerValue(String key, boolean required) {
        return getGenericValue(key, Integer.class, required, (obj) -> Integer.valueOf(obj.toString()));
    }

    public Long getLongValue(String key, boolean required) {
        return getGenericValue(key, Long.class, required, (obj) -> Longs.tryParse(obj.toString()));
    }

    public RouteDirection getDirection(boolean required) {
        var directionValue = getIntegerValue(Headers.DIRECTION, required);
        if (Objects.isNull(directionValue)) {
            return null;
        }
        return RouteDirection.valueOf(directionValue);
    }

    public MessageHeader setDirection(RouteDirection direction) {
        Preconditions.checkNotNull(direction, "RouteDirection can not be null!");
        put(Headers.DIRECTION, direction.getValue());
        return this;
    }

    public String getCommandType(boolean required) {
        return getStringValue(Headers.COMMAND_TYPE, required);
    }

    public MessageHeader setCommandType(String commandType) {
        Preconditions.checkNotNull(commandType, "commandType can not be null!");
        put(Headers.COMMAND_TYPE, commandType);
        return this;
    }

    public Long getId(boolean required) {
        return getLongValue(Headers.ID, required);
    }

    public MessageHeader setId(Long id) {
        Preconditions.checkNotNull(id, "id can not be null!");
        put(Headers.ID, id);
        return this;
    }

    public MessageHeader setSendMode(SendMode sendMode) {
        put(Headers.SEND_MODE, sendMode.getValue());
        return this;
    }
    public MessageHeader setSendModeIfAbsent(SendMode sendMode) {
        if (containsKey(Headers.SEND_MODE)){
            return this;
        }
        put(Headers.SEND_MODE, sendMode.getValue());
        return this;
    }


    public SendMode getSendMode(boolean required) {
        var sendModeValue = getIntegerValue(Headers.SEND_MODE, required);
        return SendMode.valueOf(sendModeValue);
    }

    //region CLIENT | SERVER  ACK

    /**
     * 用于 服务端 发送 ACK 消息 给客户端
     *
     * @param required
     * @return
     */
    public String getClientMessageId(boolean required) {
        return getStringValue(Headers.CLIENT_MESSAGE_ID, required);
    }

    public MessageHeader setClientMessageId(String clientMessageId) {
        Preconditions.checkNotNull(clientMessageId, "clientMessageId can not be null!");
        put(Headers.CLIENT_MESSAGE_ID, clientMessageId);
        return this;
    }

    /**
     * 用于 客户端 发送 ACK 消息 给服务端
     *
     * @param required
     * @return
     */
    public Long getServerMessageId(boolean required) {
        return getLongValue(Headers.SERVER_MESSAGE_ID, required);
    }

    public MessageHeader setServerMessageId(Long serverMessageId) {
        Preconditions.checkNotNull(serverMessageId, "serverMessageId can not be null!");
        put(Headers.SERVER_MESSAGE_ID, serverMessageId);
        return this;
    }

    //endregion

    //region SENDER

    public Integer getSenderConnectorId(boolean required) {
        return getIntegerValue(Headers.SENDER_CONNECTOR_ID, required);
    }

    public MessageHeader setSenderConnectorId(Integer senderConnectorId) {
        Preconditions.checkNotNull(senderConnectorId, "senderConnectorId can not be null!");
        put(Headers.SENDER_CONNECTOR_ID, senderConnectorId);
        return this;
    }

    public Long getSenderUserId(boolean required) {
        return getLongValue(Headers.SENDER_USER_ID, required);
    }

    public MessageHeader setSenderUserId(Long senderUserId) {
        Preconditions.checkNotNull(senderUserId, "senderUserId can not be null!");
        put(Headers.SENDER_USER_ID, senderUserId);
        return this;
    }

    public String getSenderChannelId(boolean required) {
        return getStringValue(Headers.SENDER_CHANNEL_ID, required);
    }

    public MessageHeader setSenderChannelId(String senderChannelId) {
        Preconditions.checkNotNull(senderChannelId, "senderChannelId can not be null!");
        put(Headers.SENDER_CHANNEL_ID, senderChannelId);
        return this;
    }

    public Long getSenderDeviceId(boolean required) {
        return getLongValue(Headers.SENDER_DEVICE_ID, required);
    }

    public MessageHeader setSenderDeviceId(Long senderDeviceId) {
        Preconditions.checkNotNull(senderDeviceId, "senderDeviceId can not be null!");
        put(Headers.SENDER_DEVICE_ID, senderDeviceId);
        return this;
    }

    public Long getCreateTime(boolean required) {
        return getLongValue(Headers.CREATE_TIME, required);
    }

    public MessageHeader setCreateTime(Long createTime) {
        Preconditions.checkNotNull(createTime, "createTime can not be null!");
        put(Headers.CREATE_TIME, createTime);
        return this;
    }

    /**
     * set {@link System#currentTimeMillis()} to create time
     *
     * @return
     */
    public MessageHeader ofNow() {
        setCreateTime(System.currentTimeMillis());
        return this;
    }

    //endregion

    //region RECEIVER

    public String getReceiverChannelId(boolean required) {
        return getStringValue(Headers.RECEIVER_CHANNEL_ID, required);
    }
    /**
     * set {@link Headers#RECEIVER_CHANNEL_ID} and setSendModeIfAbsent {@link SendMode#TO_CHANNEL}
     *
     * @param receiverChannelId
     * @return
     */
    public MessageHeader setReceiverChannelId(String receiverChannelId) {
        Preconditions.checkNotNull(receiverChannelId, "receiverChannelId can not be null!");
        setSendModeIfAbsent(SendMode.TO_CHANNEL);
        put(Headers.RECEIVER_CHANNEL_ID, receiverChannelId);
        return this;
    }

    public Long getReceiverDeviceId(boolean required) {
        return getLongValue(Headers.RECEIVER_DEVICE_ID, required);
    }
    /**
     * set {@link Headers#RECEIVER_DEVICE_ID} and setSendModeIfAbsent {@link SendMode#TO_DEVICE}
     *
     * @param receiverDeviceId
     * @return
     */
    public MessageHeader setReceiverDeviceId(Long receiverDeviceId) {
        Preconditions.checkNotNull(receiverDeviceId, "receiverDeviceId can not be null!");
        setSendModeIfAbsent(SendMode.TO_DEVICE);
        put(Headers.RECEIVER_DEVICE_ID, receiverDeviceId);
        return this;
    }

    public Long getReceiverUserId(boolean required) {
        return getLongValue(Headers.RECEIVER_USER_ID, required);
    }

    /**
     * set {@link Headers#RECEIVER_USER_ID} and setSendModeIfAbsent {@link SendMode#TO_USER}
     *
     * @param receiverUserId
     * @return
     */
    public MessageHeader setReceiverUserId(Long receiverUserId) {
        Preconditions.checkNotNull(receiverUserId, "receiverUserId can not be null!");
        setSendModeIfAbsent(SendMode.TO_USER);
        put(Headers.RECEIVER_USER_ID, receiverUserId);
        return this;
    }

    public Long getReceiverGroupId(boolean required) {
        return getLongValue(Headers.RECEIVER_GROUP_ID, required);
    }
    /**
     * set {@link Headers#RECEIVER_GROUP_ID} and setSendModeIfAbsent {@link SendMode#TO_GROUP}
     *
     * @param receiverGroupId
     * @return
     */
    public MessageHeader setReceiverGroupId(Long receiverGroupId) {
        Preconditions.checkNotNull(receiverGroupId, "receiverGroupId can not be null!");
        setSendModeIfAbsent(SendMode.TO_GROUP);
        put(Headers.RECEIVER_GROUP_ID, receiverGroupId);
        return this;
    }

    public Long getReceiverRoomId(boolean required) {
        return getLongValue(Headers.RECEIVER_ROOM_ID, required);
    }
    /**
     * set {@link Headers#RECEIVER_ROOM_ID} and setSendModeIfAbsent {@link SendMode#TO_ROOM}
     *
     * @param receiverRoomId
     * @return
     */
    public MessageHeader setReceiverRoomId(Long receiverRoomId) {
        Preconditions.checkNotNull(receiverRoomId, "receiverRoomId can not be null!");
        setSendModeIfAbsent(SendMode.TO_ROOM);
        put(Headers.RECEIVER_ROOM_ID, receiverRoomId);
        return this;
    }

    public Integer getReceiverConnectorId(boolean required) {
        return getIntegerValue(Headers.RECEIVER_CONNECTOR_ID, required);
    }

    public MessageHeader setReceiverConnectorId(Integer receiverConnectorId) {
        Preconditions.checkNotNull(receiverConnectorId, "receiverConnectorId can not be null!");
        put(Headers.RECEIVER_CONNECTOR_ID, receiverConnectorId);
        return this;
    }
    //endregion

    //region FILTER_EXPRESSION

    public String getFilterExpression(boolean required) {
        return getStringValue(Headers.FILTER_EXPRESSION, required);
    }

    public MessageHeader setFilterExpression(String filterExpression) {
        Preconditions.checkNotNull(filterExpression, "filterExpression can not be null!");
        put(Headers.FILTER_EXPRESSION, filterExpression);
        return this;
    }
    //endregion

    //region ERROR

    public String getErrorCode(boolean required) {
        return getStringValue(Headers.ERROR_CODE, required);
    }

    public MessageHeader setErrorCode(String errorCode) {
        Preconditions.checkNotNull(errorCode, "errorCode can not be null!");
        put(Headers.ERROR_CODE, errorCode);
        return this;
    }

    public String getErrorMsg(boolean required) {
        return getStringValue(Headers.ERROR_MSG, required);
    }

    public MessageHeader setErrorMsg(String errorMsg) {
        Preconditions.checkNotNull(errorMsg, "errorMsg can not be null!");
        put(Headers.ERROR_MSG, errorMsg);
        return this;
    }
    //endregion

    public String getClientIp(boolean required) {
        return getStringValue(Headers.CLIENT_IP, required);
    }

    public MessageHeader setClientIp(String clientIp) {
        Preconditions.checkNotNull(clientIp, "clientIp can not be null!");
        put(Headers.CLIENT_IP, clientIp);
        return this;
    }

    public Integer getRetryTimes(boolean required) {
        return getIntegerValue(Headers.RETRY_TIMES, required);
    }

    public MessageHeader setRetryTimes(Integer retryTimes) {
        Preconditions.checkNotNull(retryTimes, "retryTimes can not be null!");
        put(Headers.RETRY_TIMES, retryTimes);
        return this;
    }

    public boolean isRetry() {
        var retryTimes = getRetryTimes(false);
        return Objects.nonNull(retryTimes) && retryTimes > 0;
    }

    public String getApp(boolean required) {
        return getStringValue(Headers.APP, required);
    }

    public MessageHeader setApp(String app) {
        Preconditions.checkNotNull(app, "app can not be null!");
        put(Headers.APP, app);
        return this;
    }

    public interface Headers {
        /**
         * application id
         */
        String APP = "app";
        /**
         * {@link RouteDirection}
         */
        String DIRECTION = "d";
        /**
         * command type
         */
        String COMMAND_TYPE = "t";

        /**
         * 服务端消息 Id 全局唯一
         */
        String ID = "id";

        /**
         * 消息的创建时间戳
         * {@link System#currentTimeMillis()}
         */
        String CREATE_TIME = "c_t";

        String AUTH = "auth";
        /**
         * Filter 表达式
         */
        String FILTER_EXPRESSION = "f";
        /**
         * 发送方式
         * {@link SendMode}
         */
        String SEND_MODE = "m";

        /**
         * 客户端的消息 Id
         * 用于服务端 ACK
         */
        String CLIENT_MESSAGE_ID = "c_m_id";

        /**
         * 服务端的消息 Id
         * 用于客户端 ACK
         */
        String SERVER_MESSAGE_ID = "s_m_id";

        /**
         * 客户端的消息 Id
         * 用于服务端 ACK
         */
        String CLIENT_IP = "c_ip";

        /**
         * 当前重试次数
         */
        String RETRY_TIMES = "retry_times";
        //region Sender
        /**
         * 发送者 ConnectorId
         */
        String SENDER_CONNECTOR_ID = "s_c_id";
        /**
         * 发送者 UserId
         */
        String SENDER_USER_ID = "s_u_id";
        /**
         * 发送者 ChannelId
         */
        String SENDER_CHANNEL_ID = "s_h_id";
        /**
         * 发送者 DeviceId
         */
        String SENDER_DEVICE_ID = "s_d_id";

        //endregion

        //region Receiver
        /**
         * 接收连接器 Id
         */
        String RECEIVER_CONNECTOR_ID = "r_c_id";
        /**
         * 接收人 Id
         */
        String RECEIVER_USER_ID = "r_u_id";
        /**
         * 接收链接 Id
         */
        String RECEIVER_CHANNEL_ID = "r_h_id";
        /**
         * 接收设备 Id
         */
        String RECEIVER_DEVICE_ID = "r_d_id";
        /**
         * 接收群组 Id
         */
        String RECEIVER_GROUP_ID = "r_g_id";

        /**
         * 接收 RoomId
         */
        String RECEIVER_ROOM_ID = "r_r_id";
        //endregion

        //region ERROR
        String ERROR_CODE = "error_code";
        String ERROR_MSG = "error_msg";
        //endregion
    }
}
