package me.ahoo.pigeon.core.security.authorization.jwt;

import com.google.common.primitives.Longs;
import io.jsonwebtoken.Claims;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jwts;
import lombok.var;
import me.ahoo.pigeon.core.security.authorization.AuthorizationFailedException;
import me.ahoo.pigeon.core.security.authorization.Device;
import me.ahoo.pigeon.core.security.authorization.Room;

import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.time.Duration;
import java.util.Base64;
import java.util.Date;
import java.util.Objects;
import java.util.UUID;

/**
 * @author ahoo wang
 * Creation time: 2019/12/26 11:00
 */
public class JwtProvider {
    private final SecretKey secretKey;
    private final Duration validityTime;
    private static final String DEVICE_NAME = "name";
    private static final String ROOM_NAME = "name";
    private static final String DEVICE_AGENT = "agent";

    public JwtProvider(String algName, String secretKey, Duration validityTime) {
        if (Objects.isNull(validityTime)) {
            validityTime = Duration.ofHours(2);
        }
        this.validityTime = validityTime;
        byte[] secretKeyBytes = Base64.getDecoder().decode(secretKey);
        this.secretKey = new SecretKeySpec(secretKeyBytes, algName);
    }

    public Claims parseClaims(String token) {
        try {
            var jwtToken = token.substring(JwtConstants.TOKEN_PREFIX_LENGTH);
            return Jwts.parserBuilder().setSigningKey(secretKey).build().parseClaimsJws(jwtToken).getBody();
        } catch (Throwable e) {
            throw new AuthorizationFailedException("Auth-Jwt-0001", e.getMessage(), e);
        }
    }

    public Device parseDevice(String token) {
        var claims = parseClaims(token);
        var deviceBuilder = Device.builder();
        var deviceId = Longs.tryParse(claims.getSubject());
        deviceBuilder.id(deviceId);
        var name = claims.get(DEVICE_NAME);
        if (Objects.nonNull(name)) {
            deviceBuilder.name(name.toString());
        }
        var deviceAgent = claims.get(DEVICE_AGENT);
        if (Objects.nonNull(deviceAgent)) {
            deviceBuilder.agent(deviceAgent.toString());
        }
        return deviceBuilder.build();
    }

    public Room parseRoom(String token) {
        var claims = parseClaims(token);
        var roomBuilder = Room.builder();
        var roomId = Longs.tryParse(claims.getSubject());
        roomBuilder.id(roomId);
        var name = claims.get(ROOM_NAME);
        if (Objects.nonNull(name)) {
            roomBuilder.name(name.toString());
        }
        return roomBuilder.build();
    }

    public String parseDeviceToken(Device device) {
        try {
            Date now = new Date();
            Date validity = new Date(now.getTime() + validityTime.toMillis());
            var jwtId = UUID.randomUUID().toString();
            String jwtToken = Jwts.builder()
                    .setId(jwtId)
//                    .addClaims(claims)
                    .setSubject(device.getId().toString())
                    .claim(DEVICE_NAME, device.getName())
                    .claim(DEVICE_AGENT, device.getAgent())
                    .setIssuedAt(now)
                    .setExpiration(validity)
                    .signWith(secretKey)
                    .compact();
            return JwtConstants.TOKEN_PREFIX + jwtToken;
        } catch (ExpiredJwtException e) {
            throw new AuthorizationFailedException("Auth-Jwt-0002", e.getMessage());
        } catch (Throwable e) {
            throw new AuthorizationFailedException("Auth-Jwt-0002", e.getMessage());
        }
    }
}
