package me.ahoo.pigeon.core.security.authorization.jwt;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import com.google.common.primitives.Longs;
import lombok.var;
import me.ahoo.pigeon.core.message.Message;
import me.ahoo.pigeon.core.message.MessageHeader;
import me.ahoo.pigeon.core.security.authorization.AuthorizationSupport;
import me.ahoo.pigeon.core.security.authorization.jwt.config.UserConfig;
import me.ahoo.pigeon.core.security.SecurityContext;
import me.ahoo.pigeon.core.security.authorization.AuthorizeResult;
import me.ahoo.pigeon.core.security.authorization.User;
import me.ahoo.pigeon.core.security.authorization.UserAuthorization;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.util.Objects;

/**
 * @author ahoo wang
 * Creation time: 2019/11/18 10:53
 */
@Order(Ordered.HIGHEST_PRECEDENCE)
public class JwtUserAuthorization implements UserAuthorization, AuthorizationSupport {

    private final JwtProvider jwtProvider;
    private String claimId = "id";
    private String claimUsername = "username";
    private String claimNickname = "nickname";
    private String claimAvatar = "avatar";
    private String claimRole = "role";

    private static final String CLAIM_MAP_ID = "id";
    private static final String CLAIM_MAP_USERNAME = "username";
    private static final String CLAIM_MAP_NICKNAME = "nickname";
    private static final String CLAIM_MAP_ROLE = "role";
    private static final String CLAIM_MAP_AVATAR = "avatar";


    public JwtUserAuthorization(UserConfig userConfig) {
        jwtProvider = new JwtProvider(userConfig.getAlgName(), userConfig.getSecretKey(), userConfig.getValidityTime());
        var claimMap = userConfig.getClaimMap();
        if (Objects.nonNull(claimMap)) {
            claimId = claimMap.getOrDefault(CLAIM_MAP_ID, claimId);
            claimUsername = claimMap.getOrDefault(CLAIM_MAP_USERNAME, claimUsername);
            claimNickname = claimMap.getOrDefault(CLAIM_MAP_NICKNAME, claimNickname);
            claimAvatar = claimMap.getOrDefault(CLAIM_MAP_AVATAR, claimAvatar);
            claimRole = claimMap.getOrDefault(CLAIM_MAP_ROLE, claimRole);
        }
    }

    @Override
    public AuthorizeResult<User> authorize(SecurityContext securityContext, Message authMessage) {
        var jwtToken = authMessage.getHeader().get(MessageHeader.Headers.AUTH);
        Preconditions.checkNotNull(jwtToken, "Header.%s can not be null.", MessageHeader.Headers.AUTH);
        var claims = jwtProvider.parseClaims(jwtToken.toString());
        var userBuilder = User.builder();
        var userId = claims.get(claimId);
        Preconditions.checkNotNull(userId, "id can not be null.");
        userBuilder.id(Longs.tryParse(userId.toString()));

        var username = claims.get(claimUsername);
        if (Objects.nonNull(username)) {
            userBuilder.username(username.toString());
        }
        var nickname = claims.get(claimNickname);
        if (Objects.nonNull(nickname)) {
            userBuilder.nickname(nickname.toString());
        }

        var avatar = claims.get(claimAvatar);
        if (Objects.nonNull(avatar)) {
            userBuilder.avatar(avatar.toString());
        }
        var role = claims.get(claimRole);
        if (Objects.nonNull(role)) {
            userBuilder.role(role.toString());
        }
        userBuilder.attr(ImmutableMap.copyOf(claims));
        return AuthorizeResult.<User>builder().accessToken(jwtToken.toString()).body(userBuilder.build()).build();
    }


    @Override
    public boolean support(Message authMessage) {
        var authObj = authMessage.getHeader().get(MessageHeader.Headers.AUTH);
        if (Objects.isNull(authObj)) {
            return false;
        }
        var authStr = authObj.toString();
        return authStr.startsWith(JwtConstants.TOKEN_PREFIX);
    }


}
