package me.ahoo.pigeon.starter.security;

import lombok.var;
import me.ahoo.pigeon.core.id.IdGenerator;
import me.ahoo.pigeon.core.id.impl.SecondSnowflakeId;
import me.ahoo.pigeon.core.security.authorization.*;
import me.ahoo.pigeon.core.security.authorization.jwt.JwtDeviceAuthorization;
import me.ahoo.pigeon.core.security.authorization.jwt.JwtRoomAuthorization;
import me.ahoo.pigeon.core.security.authorization.jwt.JwtUserAuthorization;
import me.ahoo.pigeon.core.security.authorization.none.NoneDeviceAuthorization;
import me.ahoo.pigeon.core.security.authorization.none.NoneRoomAuthorization;
import me.ahoo.pigeon.core.security.authorization.none.NoneUserAuthorization;
import me.ahoo.pigeon.core.security.command.CommandAuthorization;
import me.ahoo.pigeon.core.security.command.PrivilegeCommandAuthorization;
import me.ahoo.pigeon.core.security.command.TrueCommandAuthorization;
import me.ahoo.pigeon.core.security.device.DeviceRegistrar;
import me.ahoo.pigeon.core.security.device.TemporaryDeviceRegistrar;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.List;
import java.util.Objects;

/**
 * @author ahoo wang
 * Creation time: 2020/3/18 11:43
 */
@Configuration(proxyBeanMethods = false)
@EnableConfigurationProperties({SecurityProperties.class})
public class SecurityAutoConfiguration {

    private final SecurityProperties securityProperties;

    public SecurityAutoConfiguration(SecurityProperties securityProperties) {
        this.securityProperties = securityProperties;
    }

    @Bean
    @ConditionalOnMissingBean
    public CommandAuthorization privilegeCommandAuthorization() {
        var commandSecurity = securityProperties.getCommand();
        if (Objects.isNull(commandSecurity)) {
            return TrueCommandAuthorization.INSTANCE;
        }
        var privilegeSecurityChecker = commandSecurity.asPrivilegeSecurityChecker();
        return new PrivilegeCommandAuthorization(privilegeSecurityChecker);
    }

    @Bean
    @ConditionalOnBean(AuthorizationSupport.class)
    @ConditionalOnMissingBean
    public CompositeAuthorization compositeAuthorization(List<AuthorizationSupport> delegates) {
        return new CompositeAuthorization(delegates);
    }

    @Bean
    @ConditionalOnMissingBean
    public DeviceRegistrar deviceRegistrar(SecondSnowflakeId secondSnowflakeId) {
        return new TemporaryDeviceRegistrar(secondSnowflakeId);
    }


    public class JwtAuth {
        @Bean
        @ConditionalOnMissingBean
        @ConditionalOnProperty(prefix = SecurityProperties.PREFIX, name = "auth.jwt.user.alg-name")
        public UserAuthorization userAuthorization() {
            return new JwtUserAuthorization(securityProperties.getAuth().getJwt().getUser());
        }

        @Bean
        @ConditionalOnMissingBean
        @ConditionalOnProperty(prefix = SecurityProperties.PREFIX, name = "auth.jwt.device.alg-name")
        public DeviceAuthorization deviceAuthorization(DeviceRegistrar deviceRegistrar) {
            return new JwtDeviceAuthorization(deviceRegistrar, securityProperties.getAuth().getJwt().getDevice());
        }

        @Bean
        @ConditionalOnMissingBean
        @ConditionalOnProperty(prefix = SecurityProperties.PREFIX, name = "auth.jwt.room.alg-name")
        public RoomAuthorization roomAuthorization() {
            return new JwtRoomAuthorization(securityProperties.getAuth().getJwt().getRoom());
        }
    }

    @ConditionalOnProperty(prefix = SecurityProperties.PREFIX, name = "auth.none")
    public static class NoneAuth {
        @Bean
        @ConditionalOnMissingBean
        public UserAuthorization userAuthorization(IdGenerator idGenerator) {
            return new NoneUserAuthorization(idGenerator);
        }

        @Bean
        @ConditionalOnMissingBean
        public DeviceAuthorization deviceAuthorization(DeviceRegistrar deviceRegistrar) {
            return new NoneDeviceAuthorization(deviceRegistrar);
        }

        @Bean
        @ConditionalOnMissingBean
        public RoomAuthorization roomAuthorization() {
            return new NoneRoomAuthorization();
        }
    }
}
