/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.command

import me.ahoo.wow.Version
import me.ahoo.wow.command.annotation.asCommandMetadata
import me.ahoo.wow.id.IdFactory
import me.ahoo.wow.id.generateId
import me.ahoo.wow.messaging.Header
import me.ahoo.wow.messaging.NamedMessage
import me.ahoo.wow.modeling.AggregateId
import me.ahoo.wow.modeling.NamedAggregate
import me.ahoo.wow.modeling.asAggregateId
import java.time.Clock

/**
 * Command idempotency ID.
 */
interface RequestId {
    val requestId: String
}

interface CommandId {
    val commandId: String
}

/**
 * Command Message .
 *
 * @author ahoo wang
 */
interface CommandMessage<C : Any> : NamedMessage<C>, NamedAggregate, CommandId, RequestId {
    override val commandId: String
        get() = id

    /**
     * target aggregate id
     */
    val aggregateId: AggregateId

    /**
     * expected aggregate version
     */
    val aggregateVersion: Int?

    /**
     * is the create aggregate command
     */
    val isCreate: Boolean

    override fun mergeHeader(additionalSource: Map<String, String>): CommandMessage<C>
}

data class SimpleCommandMessage<C : Any>(
    override val id: String,
    override val header: Header,
    override val body: C,
    override val aggregateId: AggregateId,
    override val requestId: String,
    override val aggregateVersion: Int?,
    override val name: String,
    override val isCreate: Boolean,
    override val createTime: Long = Clock.systemUTC().millis()
) : CommandMessage<C>, NamedAggregate by aggregateId {

    override fun mergeHeader(additionalSource: Map<String, String>): CommandMessage<C> =
        copy(header = header.mergeWith(additionalSource))
}

fun <C : Any> C.asCommandMessage(
    id: String = IdFactory.generateAsString(),
    requestId: String? = null,
    aggregateId: String? = null,
    header: Header = Header.EMPTY,
    createTime: Long = Clock.systemUTC().millis()
): CommandMessage<C> {
    val metadata = javaClass.asCommandMetadata()
    val namedAggregate = metadata.namedAggregateGetter.getNamedAggregate(this)
    val aggregateIdString = metadata.aggregateIdGetter?.get(this) ?: aggregateId ?: namedAggregate.generateId()
    val targetAggregateId = namedAggregate.asAggregateId(aggregateIdString)
    val expectedAggregateVersion = if (metadata.isCreate) {
        Version.UNINITIALIZED_VERSION
    } else {
        metadata.aggregateVersionGetter?.get(this)
    }

    return SimpleCommandMessage(
        id = id,
        requestId = requestId ?: id,
        header = header,
        body = this,
        createTime = createTime,
        aggregateId = targetAggregateId,
        aggregateVersion = expectedAggregateVersion,
        name = metadata.name,
        isCreate = metadata.isCreate
    )
}
