/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.command

import me.ahoo.wow.command.wait.CommandStage
import me.ahoo.wow.command.wait.CommandWaitEndpoint
import me.ahoo.wow.command.wait.SimpleWaitSignal
import me.ahoo.wow.command.wait.WaitStrategy
import me.ahoo.wow.command.wait.WaitStrategyRegistrar
import me.ahoo.wow.command.wait.WaitingFor
import me.ahoo.wow.command.wait.injectWaitStrategy
import me.ahoo.wow.exception.asErrorInfo
import me.ahoo.wow.infra.idempotency.IdempotencyChecker
import me.ahoo.wow.messaging.tracing.getUpstream
import me.ahoo.wow.messaging.tracing.trace
import reactor.core.publisher.Mono
import reactor.kotlin.core.publisher.toMono

class DefaultCommandGateway(
    private val commandWaitEndpoint: CommandWaitEndpoint,
    private val commandBus: CommandBus,
    private val idempotencyChecker: IdempotencyChecker,
    private val waitStrategyRegistrar: WaitStrategyRegistrar
) : CommandGateway, CommandBus by commandBus {

    override fun <C : Any> send(command: CommandMessage<C>): Mono<Void> {
        return Mono.deferContextual { context ->
            val tracedCommand = command.trace(context.getUpstream())
            commandBus.send(tracedCommand)
        }
    }

    override fun <C : Any> send(
        command: CommandMessage<C>,
        waitStrategy: WaitStrategy
    ): Mono<out ClientCommandExchange<C>> {
        return idempotencyChecker.check(command.id)
            .doOnNext {
                /*
                 * 检查命令幂等性，如果该命令通过幂等性检查则返回 {@code true},表示该命令不重复.
                 */
                if (!it) {
                    throw DuplicateCommandException(command)
                }
            }
            .flatMap { _ ->
                require(waitStrategy is WaitingFor) { "waitStrategy must be WaitingFor." }
                val awaitableCommand =
                    command.injectWaitStrategy<CommandMessage<C>>(
                        commandWaitEndpoint = commandWaitEndpoint.endpoint,
                        stage = waitStrategy.stage
                    )
                val commandExchange: ClientCommandExchange<C> =
                    SimpleClientCommandExchange(awaitableCommand, waitStrategy)
                waitStrategyRegistrar.register(awaitableCommand.commandId, waitStrategy)
                send(awaitableCommand)
                    .thenReturn(
                        SimpleWaitSignal(
                            commandId = awaitableCommand.id,
                            stage = CommandStage.SENT
                        )
                    )
                    .onErrorResume {
                        val errorInfo = it.asErrorInfo()
                        SimpleWaitSignal(
                            commandId = awaitableCommand.id,
                            stage = CommandStage.SENT,
                            errorCode = errorInfo.errorCode,
                            errorMsg = errorInfo.errorMsg
                        ).toMono()
                    }.doOnNext {
                        waitStrategy.next(it)
                    }.thenReturn(commandExchange)
            }
    }
}
