/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.command.annotation

import me.ahoo.wow.command.metadata.CommandMetadata
import me.ahoo.wow.configuration.asRequiredNamedAggregate
import me.ahoo.wow.configuration.asRequiredNamedBoundedContext
import me.ahoo.wow.infra.accessor.field.FieldAccessor
import me.ahoo.wow.infra.accessor.field.FieldGetter
import me.ahoo.wow.infra.reflection.AnnotationScanner.scan
import me.ahoo.wow.infra.reflection.ClassMetadata.visit
import me.ahoo.wow.infra.reflection.ClassVisitor
import me.ahoo.wow.metadata.CacheableMetadataParser
import me.ahoo.wow.modeling.annotation.AggregateId
import me.ahoo.wow.modeling.annotation.AggregateName
import me.ahoo.wow.modeling.matedata.MetadataNamedAggregateGetter
import me.ahoo.wow.modeling.matedata.SimpleNamedAggregateGetter
import me.ahoo.wow.naming.annotation.asName
import java.lang.reflect.Field

/**
 * Command Metadata Parser .
 *
 * @author ahoo wang
 */
object CommandMetadataParser : CacheableMetadataParser<CommandMetadata<*>>() {

    override fun parseAsMetadata(type: Class<*>): CommandMetadata<*> {
        val visitor = CommandMetadataVisitor(type)
        visit(type, visitor)
        return visitor.asMetadata()
    }
}

internal class CommandMetadataVisitor<C>(private val commandType: Class<C>) : ClassVisitor {
    private val commandName: String = commandType.asName()
    private val isCreateAggregate = commandType.isAnnotationPresent(CreateAggregate::class.java)
    private var aggregateNameField: Field? = null
    private var aggregateIdField: Field? = null
    private var aggregateVersionField: Field? = null
    override fun visitField(field: Field) {
        field.scan<AggregateName>()?.let {
            if (aggregateNameField == null) {
                aggregateNameField = field
            }
        }
        field.scan<AggregateId>()?.let {
            if (aggregateIdField == null) {
                aggregateIdField = field
            }
        }
        field.scan<AggregateVersion>()?.let {
            if (aggregateVersionField == null) {
                aggregateVersionField = field
            }
        }
    }

    fun asMetadata(): CommandMetadata<C> {
        val aggregateNameGetter: FieldGetter<C, String>? = aggregateNameField?.let {
            FieldAccessor(it)
        }
        val aggregateVersionGetter: FieldGetter<C, Int?>? = aggregateVersionField?.let {
            FieldAccessor(it)
        }

        val aggregateIdGetter: FieldGetter<C, String>? = aggregateIdField?.let {
            FieldAccessor(it)
        }
        if (aggregateIdGetter == null && !isCreateAggregate) {
            throw IllegalArgumentException("Non-create aggregation commands must define the aggregation ID field.")
        }
        val namedAggregateGetter = if (aggregateNameGetter == null) {
            MetadataNamedAggregateGetter(commandType.asRequiredNamedAggregate())
        } else {
            SimpleNamedAggregateGetter(
                contextName = commandType.asRequiredNamedBoundedContext().contextName,
                aggregateNameGetter = aggregateNameGetter
            )
        }
        return CommandMetadata(
            commandType = commandType,
            namedAggregateGetter = namedAggregateGetter,
            name = commandName,
            isCreate = isCreateAggregate,
            aggregateIdGetter = aggregateIdGetter,
            aggregateVersionGetter = aggregateVersionGetter
        )
    }
}

fun <C> Class<C>.asCommandMetadata(): CommandMetadata<C> {
    @Suppress("UNCHECKED_CAST")
    return CommandMetadataParser.parse(this) as CommandMetadata<C>
}

inline fun <reified C> commandMetadata(): CommandMetadata<C> {
    return C::class.java.asCommandMetadata()
}
