/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.event

import me.ahoo.wow.Version
import me.ahoo.wow.command.CommandId
import me.ahoo.wow.event.annotation.asEventMetadata
import me.ahoo.wow.id.IdFactory
import me.ahoo.wow.messaging.Header
import me.ahoo.wow.messaging.NamedMessage
import me.ahoo.wow.modeling.AggregateId
import me.ahoo.wow.modeling.DEFAULT_REVISION
import me.ahoo.wow.modeling.NamedAggregate
import me.ahoo.wow.modeling.Revision
import me.ahoo.wow.modeling.asAggregateId
import java.time.Clock

const val DEFAULT_EVENT_SEQUENCE = 1

/**
 * DomainEvent .
 *
 * Events published when a command is processed by the aggregate
 *
 * 由聚合发布的领域事件 .
 *
 * 领域事件推荐使用声明式（Declarative）设计的方式（幂等，类似于 Kubernetes apply、Docker 镜像层）
 * 即聚合根在事件朔源时只需要简单的将领域事件作为覆盖层（onSourcing 只对状态赋值，没有逻辑判断），
 * 事件朔源将可以不需要显式定义朔源函数。
 * @author ahoo wang
 */
interface DomainEvent<T : Any> : NamedMessage<T>, CommandId, NamedAggregate, Version, Revision {
    val aggregateId: AggregateId
    val sequence: Int
        get() = DEFAULT_EVENT_SEQUENCE
    override val revision: String
        get() = DEFAULT_REVISION

    /**
     * 是否为事件流的最后一个事件
     */
    val isLast: Boolean
        get() = true
}

data class SimpleDomainEvent<T : Any>(
    override val id: String,
    override val header: Header,
    override val body: T,
    override val aggregateId: AggregateId,
    override val version: Int,
    override val sequence: Int,
    override val revision: String = DEFAULT_REVISION,
    override val commandId: String,
    override val name: String,
    override val isLast: Boolean,
    override val createTime: Long = Clock.systemUTC().millis()
) : DomainEvent<T>, NamedAggregate by aggregateId {
    override fun mergeHeader(additionalSource: Map<String, String>): SimpleDomainEvent<T> =
        copy(header = header.mergeWith(additionalSource))
}

fun <T : Any> T.asDomainEvent(
    aggregateId: AggregateId,
    commandId: String,
    id: String = IdFactory.generateAsString(),
    version: Int = Version.INITIAL_VERSION,
    sequence: Int = DEFAULT_EVENT_SEQUENCE,
    isLast: Boolean = true,
    header: Header = Header.EMPTY,
    createTime: Long = Clock.systemUTC().millis()
): DomainEvent<T> {
    val metadata = javaClass.asEventMetadata()

    return SimpleDomainEvent(
        id = id,
        version = version,
        revision = metadata.revision,
        aggregateId = aggregateId,
        commandId = commandId,
        name = metadata.name,
        sequence = sequence,
        isLast = isLast,
        header = header,
        body = this,
        createTime = createTime
    )
}

fun <T : Any> T.asDomainEvent(
    aggregateId: String,
    commandId: String,
    id: String = IdFactory.generateAsString(),
    version: Int = Version.INITIAL_VERSION,
    sequence: Int = DEFAULT_EVENT_SEQUENCE,
    isLast: Boolean = true,
    header: Header = Header.EMPTY,
    createTime: Long = Clock.systemUTC().millis()
): DomainEvent<T> {
    val metadata = javaClass.asEventMetadata()
    val namedAggregate = metadata.namedAggregateGetter.getNamedAggregate(this)
    return SimpleDomainEvent(
        id = id,
        version = version,
        revision = metadata.revision,
        aggregateId = namedAggregate.asAggregateId(aggregateId),
        commandId = commandId,
        name = metadata.name,
        sequence = sequence,
        isLast = isLast,
        header = header,
        body = this,
        createTime = createTime
    )
}
