/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.ahoo.wow.event

import me.ahoo.wow.Version
import me.ahoo.wow.command.CommandId
import me.ahoo.wow.command.CommandMessage
import me.ahoo.wow.command.RequestId
import me.ahoo.wow.command.wait.extractWaitStrategy
import me.ahoo.wow.command.wait.injectWaitStrategy
import me.ahoo.wow.id.IdFactory
import me.ahoo.wow.messaging.Header
import me.ahoo.wow.messaging.Message
import me.ahoo.wow.messaging.NamedBoundedContextMessage
import me.ahoo.wow.modeling.AggregateId
import me.ahoo.wow.modeling.NamedAggregate
import java.time.Clock

/**
 * Event Stream .
 * Relation: `Event Stream` 1:1 `CommandId`.
 *
 * 必须保证按照版本号升序排序，且版本号单调递增.
 * @author ahoo wang
 */
interface DomainEventStream :
    NamedBoundedContextMessage<List<DomainEvent<*>>>,
    RequestId,
    CommandId,
    NamedAggregate,
    Version,
    Iterable<DomainEvent<*>> {
    val aggregateId: AggregateId
    val size: Int
}

data class SimpleDomainEventStream(
    override val id: String,
    override val requestId: String,
    override val header: Header,
    override val body: List<DomainEvent<*>>
) :
    DomainEventStream,
    Iterable<DomainEvent<*>> by body {
    override val aggregateId: AggregateId

    override val contextName: String
        get() = aggregateId.contextName
    override val aggregateName: String
        get() = aggregateId.aggregateName

    override val commandId: String
    override val version: Int
    override val size: Int
    override val createTime: Long
    override fun mergeHeader(additionalSource: Map<String, String>): Message<List<DomainEvent<*>>> =
        copy(header = header.mergeWith(additionalSource))

    init {
        require(body.isNotEmpty()) { "events can not be empty." }
        body.first().let {
            aggregateId = it.aggregateId
            commandId = it.commandId
            version = it.version
            createTime = it.createTime
        }
        size = body.size
    }
}

fun Any.asDomainEventStream(
    command: CommandMessage<*>,
    aggregateVersion: Int,
    header: Header = Header.EMPTY
): DomainEventStream {
    val awaitableHeader = command.extractWaitStrategy()?.let {
        header.injectWaitStrategy(it.commandWaitEndpoint, it.stage)
    } ?: header
    val eventStreamId = IdFactory.generateAsString()
    val aggregateId = command.aggregateId
    val streamVersion = aggregateVersion + 1
    val createTime = Clock.systemUTC().millis()

    val events = when (this) {
        is Iterable<*> -> {
            asDomainEvents(streamVersion, aggregateId, command, awaitableHeader, createTime)
        }

        is Array<*> -> {
            asDomainEvents(streamVersion, aggregateId, command, awaitableHeader, createTime)
        }

        else -> {
            asDomainEvents(streamVersion, aggregateId, command, awaitableHeader, createTime)
        }
    }

    return SimpleDomainEventStream(
        id = eventStreamId,
        requestId = command.requestId,
        header = awaitableHeader,
        body = events
    )
}

private fun Any.asDomainEvents(
    streamVersion: Int,
    aggregateId: AggregateId,
    command: CommandMessage<*>,
    tracedHeader: Header,
    createTime: Long
): List<DomainEvent<Any>> {
    val domainEvent = this.asDomainEvent(
        id = IdFactory.generateAsString(),
        version = streamVersion,
        aggregateId = aggregateId,
        commandId = command.commandId,
        header = tracedHeader,
        createTime = createTime
    )
    return listOf(domainEvent)
}

private fun Array<*>.asDomainEvents(
    streamVersion: Int,
    aggregateId: AggregateId,
    command: CommandMessage<*>,
    tracedHeader: Header,
    createTime: Long
) = mapIndexed { index, event ->
    val sequence = (index + DEFAULT_EVENT_SEQUENCE)
    event!!.asDomainEvent(
        id = IdFactory.generateAsString(),
        version = streamVersion,
        sequence = sequence,
        isLast = sequence == this.size,
        aggregateId = aggregateId,
        commandId = command.commandId,
        header = tracedHeader,
        createTime = createTime
    )
}.toList()

private fun Iterable<*>.asDomainEvents(
    streamVersion: Int,
    aggregateId: AggregateId,
    command: CommandMessage<*>,
    tracedHeader: Header,
    createTime: Long
): List<DomainEvent<Any>> {
    val eventCount = count()
    return mapIndexed { index, event ->
        val sequence = (index + DEFAULT_EVENT_SEQUENCE)
        event!!.asDomainEvent(
            id = IdFactory.generateAsString(),
            version = streamVersion,
            sequence = sequence,
            isLast = sequence == eventCount,
            aggregateId = aggregateId,
            commandId = command.commandId,
            header = tracedHeader,
            createTime = createTime
        )
    }.toList()
}
