/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.event.annotation

import me.ahoo.wow.configuration.asRequiredNamedAggregate
import me.ahoo.wow.configuration.asRequiredNamedBoundedContext
import me.ahoo.wow.event.metadata.EventMetadata
import me.ahoo.wow.infra.accessor.field.FieldAccessor
import me.ahoo.wow.infra.accessor.field.FieldGetter
import me.ahoo.wow.infra.reflection.AnnotationScanner.scan
import me.ahoo.wow.infra.reflection.ClassMetadata
import me.ahoo.wow.infra.reflection.ClassVisitor
import me.ahoo.wow.metadata.CacheableMetadataParser
import me.ahoo.wow.modeling.DEFAULT_REVISION
import me.ahoo.wow.modeling.annotation.AggregateId
import me.ahoo.wow.modeling.annotation.AggregateName
import me.ahoo.wow.modeling.matedata.MetadataNamedAggregateGetter
import me.ahoo.wow.modeling.matedata.SimpleNamedAggregateGetter
import me.ahoo.wow.naming.annotation.asName
import java.lang.reflect.Field

/**
 * Event Metadata Parser .
 *
 * @author ahoo wang
 */
object EventMetadataParser : CacheableMetadataParser<EventMetadata<*>>() {

    override fun parseAsMetadata(type: Class<*>): EventMetadata<*> {
        val visitor = EventMetadataVisitor(type)
        ClassMetadata.visit(type, visitor)
        return visitor.asMetadata()
    }

    internal class EventMetadataVisitor<E>(private val eventType: Class<E>) : ClassVisitor {
        private val eventName: String = eventType.asName()
        var aggregateNameField: Field? = null
        private var revision = DEFAULT_REVISION
        private var integration = false
        var aggregateId: Field? = null

        init {
            val event = eventType.getAnnotation(Event::class.java)
            event?.let {
                if (it.revision.isNotEmpty()) {
                    revision = it.revision
                }
                integration = it.integration
            }
        }

        override fun visitField(field: Field) {
            field.scan<AggregateName>()?.let {
                if (aggregateNameField == null) {
                    aggregateNameField = field
                }
            }
            field.scan<AggregateId>()?.let {
                if (aggregateId == null) {
                    aggregateId = field
                }
            }
        }

        fun asMetadata(): EventMetadata<E> {
            val aggregateNameGetter: FieldGetter<E, String>? = aggregateNameField?.let {
                FieldAccessor(it)
            }
            val namedAggregateGetter = if (aggregateNameGetter == null) {
                MetadataNamedAggregateGetter(eventType.asRequiredNamedAggregate())
            } else {
                SimpleNamedAggregateGetter(
                    contextName = eventType.asRequiredNamedBoundedContext().contextName,
                    aggregateNameGetter = aggregateNameGetter
                )
            }
            return EventMetadata(
                eventType = eventType,
                namedAggregateGetter = namedAggregateGetter,
                name = eventName,
                revision = revision,
                isIntegration = integration,
                aggregateIdGetter = aggregateId?.let { FieldAccessor(it) }
            )
        }
    }
}

fun <E> Class<E>.asEventMetadata(): EventMetadata<E> {
    @Suppress("UNCHECKED_CAST")
    return EventMetadataParser.parse(this) as EventMetadata<E>
}

inline fun <reified E> eventMetadata(): EventMetadata<E> {
    return E::class.java.asEventMetadata()
}
