/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.event.serialization

import com.fasterxml.jackson.core.type.TypeReference
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.node.ArrayNode
import me.ahoo.wow.event.DomainEventStream
import me.ahoo.wow.event.SimpleDomainEventStream
import me.ahoo.wow.messaging.DefaultHeader
import me.ahoo.wow.messaging.Header
import me.ahoo.wow.messaging.serialization.EventStreamJsonDeserializer
import me.ahoo.wow.messaging.serialization.JsonSerializer
import me.ahoo.wow.messaging.serialization.asEventStreamRecord
import me.ahoo.wow.messaging.serialization.asJsonString
import java.util.*

val HEADER_TYPE = object : TypeReference<HashMap<String, String>>() {}

object JsonEventStreamSerializer {
    private fun deserializeHeader(header: String): Header {
        val headerValues = JsonSerializer.readValue(header, HEADER_TYPE)
        return DefaultHeader(headerValues)
    }

    fun serialize(eventStream: DomainEventStream): DomainEventStreamRecord {
        val eventStreamNode = JsonSerializer.valueToTree<JsonNode>(eventStream)
            .asEventStreamRecord()

        return DomainEventStreamRecord(
            id = eventStream.id,
            aggregateId = eventStream.aggregateId,
            requestId = eventStream.requestId,
            commandId = eventStream.commandId,
            version = eventStream.version,
            header = eventStreamNode.rawHeader.asJsonString(),
            body = eventStreamNode.body.asJsonString(),
            createTime = eventStream.createTime
        )
    }

    fun deserialize(eventStreamRecord: DomainEventStreamRecord): DomainEventStream {
        val eventStreamBody = JsonSerializer.readValue(eventStreamRecord.body, ArrayNode::class.java)
        val header = deserializeHeader(eventStreamRecord.header)
        val events = EventStreamJsonDeserializer.deserializeEvents(
            eventStreamBody = eventStreamBody,
            aggregateId = eventStreamRecord.aggregateId,
            version = eventStreamRecord.version,
            commandId = eventStreamRecord.commandId,
            header = header,
            createTime = eventStreamRecord.createTime
        )
        return SimpleDomainEventStream(
            id = eventStreamRecord.id,
            requestId = eventStreamRecord.requestId,
            header = header,
            body = events
        )
    }
}
