/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.eventsourcing.snapshot

import me.ahoo.wow.messaging.serialization.asJsonString
import me.ahoo.wow.messaging.serialization.asObject
import me.ahoo.wow.modeling.AggregateId
import me.ahoo.wow.modeling.matedata.StateAggregateMetadata
import reactor.core.publisher.Mono
import java.util.*
import java.util.concurrent.ConcurrentHashMap

class InMemorySnapshotRepository : SnapshotRepository {
    private val aggregateIdMapSnapshots = ConcurrentHashMap<AggregateId, LinkedList<String>>()

    override fun <S : Any> load(metadata: StateAggregateMetadata<S>, aggregateId: AggregateId): Mono<Snapshot<S>> {
        return Mono.fromCallable {
            val snapshots = aggregateIdMapSnapshots[aggregateId]
            if (snapshots == null) {
                null
            } else {
                @Suppress("UNCHECKED_CAST")
                snapshots.last.asObject(Snapshot::class.java) as Snapshot<S>
            }
        }
    }

    override fun <S : Any> load(
        metadata: StateAggregateMetadata<S>,
        aggregateId: AggregateId,
        version: Int
    ): Mono<Snapshot<S>> {
        return Mono.fromCallable {
            val snapshots = aggregateIdMapSnapshots[aggregateId]?.map {
                @Suppress("UNCHECKED_CAST")
                it.asObject(Snapshot::class.java) as Snapshot<Any>
            } ?: return@fromCallable null
            val snapshot = snapshots.singleOrNull {
                it.version == version
            }
            @Suppress("UNCHECKED_CAST")
            snapshot as Snapshot<S>
        }
    }

    @Suppress("UNCHECKED_CAST")
    override fun <S : Any> append(snapshot: Snapshot<S>): Mono<Void> {
        return Mono.fromRunnable {
            aggregateIdMapSnapshots.compute(snapshot.aggregateId) { _, value ->
                if (value == null) {
                    val snapshots = LinkedList<String>()
                    snapshots.add(snapshot.asJsonString())
                    return@compute snapshots
                }

                val snapshots = value.map {
                    it.asObject(Snapshot::class.java) as Snapshot<Any>
                }

                snapshots.any {
                    it.version == snapshot.version
                }.let {
                    if (it) {
                        throw IllegalStateException("Snapshot already exists! aggregateId:${snapshot.aggregateId},version:${snapshot.version}")
                    }
                }
                value.add(snapshot.asJsonString())
                value
            }
        }
    }
}
