/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.ahoo.wow.id

import me.ahoo.cosid.CosId
import me.ahoo.cosid.cosid.CosIdGenerator
import me.ahoo.cosid.cosid.CosIdIdStateParser
import me.ahoo.cosid.cosid.CosIdState
import me.ahoo.cosid.provider.DefaultIdGeneratorProvider
import me.ahoo.wow.exception.WowException
import me.ahoo.wow.infra.Decorator
import org.slf4j.LoggerFactory
import java.util.ServiceLoader

/**
 * IdFactory is Global.
 */
object IdFactory : CosIdGenerator, Decorator<CosIdGenerator> {
    private val log = LoggerFactory.getLogger(IdFactory::class.java)
    const val ID_KEY = "wow.cosid"
    val ID_NAME: String = System.getProperty(ID_KEY, CosId.COSID)

    override val delegate: CosIdGenerator by lazy {
        val idGenOp = DefaultIdGeneratorProvider.INSTANCE.get(ID_NAME)
        if (idGenOp.isPresent) {
            val idGenerator = idGenOp.get()
            if (log.isInfoEnabled) {
                log.info("Setup $idGenerator from DefaultIdGeneratorProvider[$ID_NAME] to IdFactory.")
            }
            return@lazy idGenerator as CosIdGenerator
        }
        ServiceLoader.load(IdFactoryInitializer::class.java).forEach {
            if (log.isInfoEnabled) {
                log.info("Load $it to initialize IdFactory.")
            }
            val idGenerator = it.createIdGenerator()
            if (log.isInfoEnabled) {
                log.info("Setup $idGenerator to IdFactory.")
            }
            return@lazy idGenerator
        }
        throw NotInitializedIdGeneratorError()
    }

    override fun getMachineId(): Int {
        return delegate.machineId
    }

    override fun getLastTimestamp(): Long {
        return delegate.lastTimestamp
    }

    override fun getStateParser(): CosIdIdStateParser {
        return delegate.stateParser
    }

    override fun generateAsState(): CosIdState {
        return delegate.generateAsState()
    }
}

class NotInitializedIdGeneratorError : WowException("Id-428", "CosIdGenerator:[${IdFactory.ID_NAME}] not initialized.")
