/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.serialization

import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.deser.std.StdDeserializer
import me.ahoo.wow.event.DomainEvent
import me.ahoo.wow.event.SimpleDomainEvent
import me.ahoo.wow.infra.TypeNameMapper.asType
import me.ahoo.wow.modeling.MaterializedNamedAggregate
import me.ahoo.wow.modeling.asAggregateId

object DomainEventRecords {
    const val SEQUENCE = "sequence"
    const val REVISION = "revision"
    const val IS_LAST = "isLast"
}

interface DomainEventRecord :
    EventRecord,
    NamedBoundedContextMessageRecord,
    MessageAggregateNameRecord,
    MessageAggregateIdRecord,
    MessageCommandIdRecord,
    MessageVersionRecord {
    val sequence: Int
        get() = actual[DomainEventRecords.SEQUENCE].asInt()
    val isLast: Boolean
        get() = actual[DomainEventRecords.IS_LAST].asBoolean()
}

class DelegatingDomainEventRecord(override val actual: JsonNode) : DomainEventRecord

fun JsonNode.asDomainEventRecord(): DomainEventRecord {
    return DelegatingDomainEventRecord(this)
}

object DomainEventJsonSerializer : MessageSerializer<DomainEvent<*>>(DomainEvent::class.java) {

    override fun writeExtendedInfo(generator: JsonGenerator, value: DomainEvent<*>) {
        generator.writeStringField(MessageRecords.AGGREGATE_ID, value.aggregateId.id)
        generator.writeStringField(MessageRecords.COMMAND_ID, value.commandId)
        generator.writeNumberField(MessageRecords.VERSION, value.version)
        generator.writeNumberField(DomainEventRecords.SEQUENCE, value.sequence)
        generator.writeStringField(DomainEventRecords.REVISION, value.revision)
        generator.writeBooleanField(DomainEventRecords.IS_LAST, value.isLast)
    }
}

object DomainEventJsonDeserializer : StdDeserializer<DomainEvent<*>>(DomainEvent::class.java) {
    override fun deserialize(p: JsonParser, ctxt: DeserializationContext): DomainEvent<*> {
        val domainEventRecord = JsonSerializer.readTree<JsonNode>(p).asDomainEventRecord()
        val contextName = domainEventRecord.contextName
        val aggregateName = domainEventRecord.aggregateName
        val aggregateId = MaterializedNamedAggregate(contextName, aggregateName)
            .asAggregateId(domainEventRecord.aggregateId)
        val bodyType = domainEventRecord.bodyType.asType<Any>()
        return SimpleDomainEvent(
            id = domainEventRecord.id,
            header = domainEventRecord.header,
            body = domainEventRecord.body.asObject(bodyType),
            aggregateId = aggregateId,
            version = domainEventRecord.version,
            sequence = domainEventRecord.sequence,
            isLast = domainEventRecord.isLast,
            revision = domainEventRecord.revision,
            commandId = domainEventRecord.commandId,
            name = domainEventRecord.name,
            createTime = domainEventRecord.createTime
        )
    }
}
