/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.serialization

import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.deser.std.StdDeserializer
import me.ahoo.wow.event.DEFAULT_EVENT_SEQUENCE
import me.ahoo.wow.event.DomainEventStream
import me.ahoo.wow.event.SimpleDomainEvent
import me.ahoo.wow.event.SimpleDomainEventStream
import me.ahoo.wow.infra.TypeNameMapper.asType
import me.ahoo.wow.messaging.Header
import me.ahoo.wow.modeling.AggregateId
import me.ahoo.wow.modeling.MaterializedNamedAggregate
import me.ahoo.wow.modeling.asAggregateId

interface EventStreamRecord :
    NamedBoundedContextMessageRecord,
    MessageCommandIdRecord,
    MessageVersionRecord,
    MessageRequestIdRecord,
    MessageAggregateIdRecord,
    MessageAggregateNameRecord

class DelegatingEventStreamRecord(override val actual: JsonNode) : EventStreamRecord

fun JsonNode.asEventStreamRecord(): EventStreamRecord {
    return DelegatingEventStreamRecord(this)
}

interface EventRecord : MessageIdRecord, MessageBodyTypeRecord, MessageBodyRecord, MessageNameRecord {
    val revision: String
        get() = actual[DomainEventRecords.REVISION].asText()
}

class DelegatingEventRecord(override val actual: JsonNode) : EventRecord

fun JsonNode.asEventRecord(): EventRecord {
    return DelegatingEventRecord(this)
}

object EventStreamJsonSerializer : MessageSerializer<DomainEventStream>(DomainEventStream::class.java) {

    override fun writeExtendedInfo(generator: JsonGenerator, value: DomainEventStream) {
        generator.writeStringField(MessageRecords.AGGREGATE_ID, value.aggregateId.id)
        generator.writeStringField(MessageRecords.COMMAND_ID, value.commandId)
        generator.writeStringField(MessageRecords.REQUEST_ID, value.requestId)
        generator.writeNumberField(MessageRecords.VERSION, value.version)
    }

    override fun writeBodyType(generator: JsonGenerator, value: DomainEventStream) = Unit

    override fun writeBody(generator: JsonGenerator, value: DomainEventStream) {
        generator.writeFieldName(MessageRecords.BODY)
        generator.writeStartArray()
        value.body.forEach {
            generator.writeStartObject()
            generator.writeStringField(MessageRecords.ID, it.id)
            generator.writeStringField(MessageRecords.NAME, it.name)
            generator.writeStringField(DomainEventRecords.REVISION, it.revision)
            generator.writeStringField(MessageRecords.BODY_TYPE, it.body.javaClass.name)
            generator.writePOJOField(MessageRecords.BODY, it.body)
            generator.writeEndObject()
        }
        generator.writeEndArray()
    }
}

object EventStreamJsonDeserializer : StdDeserializer<DomainEventStream>(DomainEventStream::class.java) {
    override fun deserialize(p: JsonParser, ctxt: DeserializationContext): DomainEventStream {
        val eventStreamRecord = JsonSerializer.readTree<JsonNode>(p).asEventStreamRecord()
        val id = eventStreamRecord.id
        val commandId = eventStreamRecord.commandId
        val requestId = eventStreamRecord.requestId
        val version = eventStreamRecord.version
        val header = eventStreamRecord.header
        val createTime = eventStreamRecord.createTime
        val contextName = eventStreamRecord.contextName
        val aggregateName = eventStreamRecord.aggregateName
        val aggregateId = MaterializedNamedAggregate(contextName, aggregateName)
            .asAggregateId(eventStreamRecord.aggregateId)
        val events = deserializeEvents(
            eventStreamBody = eventStreamRecord.body,
            aggregateId = aggregateId,
            version = version,
            commandId = commandId,
            header = header,
            createTime = createTime
        )
        return SimpleDomainEventStream(
            id = id,
            requestId = requestId,
            header = header,
            body = events
        )
    }

    fun deserializeEvents(
        eventStreamBody: JsonNode,
        aggregateId: AggregateId,
        version: Int,
        commandId: String,
        header: Header,
        createTime: Long
    ): List<SimpleDomainEvent<Any>> {
        val eventCount = eventStreamBody.size()
        return eventStreamBody.mapIndexed { index, eventNode ->
            val sequence = (index + DEFAULT_EVENT_SEQUENCE)
            val eventRecord = eventNode.asEventRecord()
            val eventType = eventRecord.bodyType.asType<Any>()
            val eventBody = eventRecord.body.asObject(eventType)
            SimpleDomainEvent(
                id = eventRecord.id,
                version = version,
                sequence = sequence,
                isLast = sequence == eventCount,
                revision = eventRecord.revision,
                aggregateId = aggregateId,
                commandId = commandId,
                name = eventRecord.name,
                header = header,
                body = eventBody,
                createTime = createTime
            )
        }.toList()
    }
}
