/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.serialization

import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.SerializerProvider
import com.fasterxml.jackson.databind.deser.std.StdDeserializer
import com.fasterxml.jackson.databind.ser.std.StdSerializer
import me.ahoo.wow.eventsourcing.snapshot.SimpleSnapshot
import me.ahoo.wow.eventsourcing.snapshot.Snapshot
import me.ahoo.wow.infra.TypeNameMapper.asType
import me.ahoo.wow.modeling.MaterializedNamedAggregate
import me.ahoo.wow.modeling.annotation.asStateAggregateMetadata
import me.ahoo.wow.modeling.asAggregateId
import me.ahoo.wow.modeling.state.StateAggregate.Companion.asStateAggregate

object SnapshotRecords {
    const val STATE_TYPE: String = "stateType"
    const val STATE: String = "state"
    const val SNAPSHOT_TIME: String = "snapshotTime"
}

object SnapshotSerializer : StdSerializer<Snapshot<*>>(Snapshot::class.java) {
    override fun serialize(value: Snapshot<*>, generator: JsonGenerator, provider: SerializerProvider) {
        generator.writeStartObject()
        generator.writeStringField(MessageRecords.CONTEXT_NAME, value.aggregateId.contextName)
        generator.writeStringField(MessageRecords.AGGREGATE_NAME, value.aggregateId.aggregateName)
        generator.writeStringField(MessageRecords.AGGREGATE_ID, value.aggregateId.id)
        generator.writeNumberField(MessageRecords.VERSION, value.version)
        generator.writePOJOField(SnapshotRecords.STATE_TYPE, value.stateRoot.javaClass.name)
        generator.writePOJOField(SnapshotRecords.STATE, value.stateRoot)
        generator.writePOJOField(SnapshotRecords.SNAPSHOT_TIME, value.snapshotTime)
        generator.writeEndObject()
    }
}

object SnapshotDeserializer : StdDeserializer<Snapshot<*>>(Snapshot::class.java) {
    override fun deserialize(p: JsonParser, ctxt: DeserializationContext): Snapshot<*> {
        val snapshotRecord = JsonSerializer.readTree<JsonNode>(p)
        val metadata = snapshotRecord[SnapshotRecords.STATE_TYPE].asText().asType<Any>()
            .asStateAggregateMetadata()
        val version = snapshotRecord[MessageRecords.VERSION].asInt()
        val snapshotTime = snapshotRecord[SnapshotRecords.SNAPSHOT_TIME].asLong()
        val stateRoot = snapshotRecord[SnapshotRecords.STATE].asObject(metadata.aggregateType)
        val namedAggregate = MaterializedNamedAggregate(
            snapshotRecord[MessageRecords.CONTEXT_NAME].asText(),
            snapshotRecord[MessageRecords.AGGREGATE_NAME].asText()
        )
        val aggregateId = namedAggregate.asAggregateId(snapshotRecord[MessageRecords.AGGREGATE_ID].asText())
        val stateAggregate = metadata.asStateAggregate(aggregateId, stateRoot, version)
        return SimpleSnapshot(stateAggregate, snapshotTime)
    }
}
