/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.serialization

import com.fasterxml.jackson.core.JsonGenerator
import com.fasterxml.jackson.core.JsonParser
import com.fasterxml.jackson.databind.DeserializationContext
import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.deser.std.StdDeserializer
import me.ahoo.wow.messaging.tracing.ImmutableSpan
import me.ahoo.wow.messaging.tracing.SimpleTracingMessage
import me.ahoo.wow.messaging.tracing.TracingMessage

interface TracingRecord : NamedBoundedContextMessageRecord

class DelegatingTracingRecord(override val actual: JsonNode) : TracingRecord

fun JsonNode.asTracingRecord(): TracingRecord {
    return DelegatingTracingRecord(this)
}

object TracingJsonSerializer : MessageSerializer<TracingMessage>(TracingMessage::class.java) {
    override fun writeBodyType(generator: JsonGenerator, value: TracingMessage) = Unit
}

object TracingJsonDeserializer : StdDeserializer<TracingMessage>(TracingMessage::class.java) {
    override fun deserialize(p: JsonParser, ctxt: DeserializationContext): TracingMessage {
        val tracingRecord = JsonSerializer.readTree<JsonNode>(p)
            .asTracingRecord()
        return SimpleTracingMessage(
            id = tracingRecord.id,
            header = tracingRecord.header,
            body = tracingRecord.body.asObject<ImmutableSpan>(),
            createTime = tracingRecord.createTime,
            contextName = tracingRecord.contextName
        )
    }
}
