/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.tracing

import me.ahoo.wow.messaging.Header
import me.ahoo.wow.messaging.Message
import me.ahoo.wow.messaging.NamedBoundedContextMessage

/**
 * 消息链路追踪器.
 *
 * [《Dapper, a Large-Scale Distributed Systems Tracing Infrastructure》](https://storage.googleapis.com/pub-tools-public-publication-data/pdf/36356.pdf)
 *
 * @author ahoo wang
 */
interface MessageTracer {
    companion object {
        @JvmField
        val DEFAULT: MessageTracer = DefaultMessageTracer
    }

    fun trace(upstreamMessage: Message<*>): Map<String, String>

    fun trace(header: Header, upstreamMessage: Message<*>): Header {
        val traced = trace(upstreamMessage)
        return header.mergeWith(traced)
    }

    fun <M : Message<*>> trace(message: M, upstreamMessage: Message<*>?): M
}

object DefaultMessageTracer : MessageTracer {

    override fun trace(upstreamMessage: Message<*>): Map<String, String> {
        return buildMap {
            upstreamMessage.userId?.let {
                this[TRACE_USER_ID] = it
            }
            upstreamMessage.deviceId?.let {
                this[TRACE_DEVICE_ID] = it
            }
            upstreamMessage.sessionId?.let {
                this[TRACE_SESSION_ID] = it
            }

            if (upstreamMessage is NamedBoundedContextMessage<*>) {
                this[TRACE_UPSTREAM_CONTEXT] = upstreamMessage.contextName
            }

            val upstreamId = upstreamMessage.id
            this[TRACE_UPSTREAM_ID] = upstreamId
            this[TRACE_ID] = upstreamMessage.traceId ?: upstreamId
        }
    }

    override fun <M : Message<*>> trace(message: M, upstreamMessage: Message<*>?): M {
        val tracedHeader = upstreamMessage?.let { trace(it) } ?: mapOf(TRACE_ID to message.id)
        @Suppress("UNCHECKED_CAST")
        return message.mergeHeader(tracedHeader) as M
    }
}

fun <M : Message<*>> M.trace(
    upstreamMessage: Message<*>? = null,
    messageTracer: MessageTracer = MessageTracer.DEFAULT
): M {
    return messageTracer.trace(this, upstreamMessage)
}
