/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.tracing

import me.ahoo.wow.annotation.Component
import me.ahoo.wow.annotation.ORDER_DEFAULT_STEP
import me.ahoo.wow.annotation.ORDER_FIRST
import me.ahoo.wow.annotation.Order
import me.ahoo.wow.configuration.asRequiredNamedBoundedContext
import me.ahoo.wow.infra.reflection.AnnotationScanner.scan
import me.ahoo.wow.messaging.function.MessageFunction
import me.ahoo.wow.messaging.handler.Filter
import me.ahoo.wow.messaging.handler.FilterChain
import me.ahoo.wow.messaging.handler.MessageExchange
import reactor.core.publisher.Mono

@Order(ORDER_FIRST + ORDER_DEFAULT_STEP * 2)
class TracingFunctionFilter<T : MessageExchange<*>>(
    private val tracingGateway: TracingGateway
) : Filter<T> {
    override fun filter(exchange: T, chain: FilterChain<T>): Mono<Void> {
        return Mono.deferContextual { context ->
            val function = context.get(MessageFunction::class.java)
            val processorType = function.processor.javaClass
            val boundedContext = processorType.asRequiredNamedBoundedContext()
            val component = processorType.scan<Component>()
            checkNotNull(component) {
                "MessageFunction[$this] does not define component annotation."
            }
            val name = "${processorType.simpleName}.${function.supportedType.simpleName}"
            tracingGateway.trace(
                contextName = boundedContext.contextName,
                component = component.value,
                name = name,
                upstreamMessage = exchange.message
            ) {
                chain.filter(exchange).writeUpstream(it)
            }
        }
    }
}
