/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.tracing

import me.ahoo.wow.command.CommandId
import me.ahoo.wow.messaging.Header
import me.ahoo.wow.messaging.Message
import me.ahoo.wow.messaging.MessageGateway
import me.ahoo.wow.messaging.serialization.MessageRecords
import reactor.core.publisher.Mono
import reactor.core.scheduler.Schedulers
import java.time.Clock

interface TracingGateway : MessageGateway, TracingBus {
    fun sendAndForget(tracingMessage: TracingMessage) {
        send(tracingMessage)
            .subscribeOn(Schedulers.boundedElastic())
            .subscribe()
    }
}

class DefaultTracingGateway(
    private val tracingBus: TracingBus
) : TracingGateway, TracingBus by tracingBus

fun <T> TracingGateway.trace(
    contextName: String,
    component: String,
    name: String,
    header: Header = Header.EMPTY,
    upstreamMessage: Message<*>?,
    actual: (TracingMessage) -> Mono<T>
): Mono<T> {
    return Mono.defer {
        val span = MutableSpan(
            component = component,
            name = name,
            startTime = Clock.systemUTC().millis()
        )
        if (upstreamMessage is CommandId) {
            span.tags[MessageRecords.COMMAND_ID] = upstreamMessage.commandId
        }

        val tracingMessage =
            span.asTracingMessage(
                contextName = contextName,
                header = header,
                createTime = span.startTime
            ).trace(upstreamMessage)

        try {
            return@defer actual.invoke(tracingMessage)
                .doOnSuccess {
                    span.success()
                    sendAndForget(tracingMessage)
                }
                .doOnError {
                    span.error(it)
                    sendAndForget(tracingMessage)
                }
        } catch (e: Throwable) {
            span.error(e)
            sendAndForget(tracingMessage)
            throw e
        }
    }
}
