/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.messaging.tracing

import me.ahoo.wow.exception.ErrorCodes
import me.ahoo.wow.exception.ErrorInfo
import me.ahoo.wow.exception.asErrorInfo
import me.ahoo.wow.id.IdFactory
import me.ahoo.wow.messaging.Header
import me.ahoo.wow.messaging.Message
import me.ahoo.wow.messaging.NamedBoundedContextMessage
import java.time.Clock

const val EXTERNAL_TRACE_PREFIX = "wow-"

const val TRACE_USER_ID = "userId"
const val TRACE_DEVICE_ID = "deviceId"
const val TRACE_SESSION_ID = "sessionId"
const val TRACE_ID = "traceId"
const val TRACE_UPSTREAM_CONTEXT = "upstreamContext"
const val TRACE_UPSTREAM_ID = "upstreamId"

val Header.userId: String?
    get() = this[TRACE_USER_ID]
val Header.deviceId: String?
    get() = this[TRACE_DEVICE_ID]
val Header.sessionId: String?
    get() = this[TRACE_SESSION_ID]

val Header.traceId: String?
    get() = this[TRACE_ID]
val Header.upstreamContext: String?
    get() = this[TRACE_UPSTREAM_CONTEXT]
val Header.upstreamId: String?
    get() = this[TRACE_UPSTREAM_ID]

val Message<*>.userId: String?
    get() = header.userId
val Message<*>.deviceId: String?
    get() = header.deviceId
val Message<*>.sessionId: String?
    get() = header.sessionId

val Message<*>.traceId: String?
    get() = header.traceId
val Message<*>.upstreamContext: String?
    get() = header.upstreamContext
val Message<*>.upstreamId: String?
    get() = header.upstreamId

interface TracingMessage : NamedBoundedContextMessage<Span>, ErrorInfo {

    override val errorCode: String
        get() = body.errorCode
    override val errorMsg: String
        get() = body.errorMsg
}

interface Span : ErrorInfo {
    /**
     * [me.ahoo.wow.annotation.Components]
     */
    val component: String

    /**
     * name/operationName:
     *
     * - aggregate: `aggregateName.commandName`
     * - handler: `className.messageName`
     */
    val name: String
    val startTime: Long
    val endTime: Long
    val duration: Long
        get() = endTime - startTime
    val tags: Map<String, String>
    override val errorCode: String
        get() = ErrorCodes.SUCCEEDED
    override val errorMsg: String
        get() = ErrorCodes.SUCCEEDED_MSG
    val stackTrace: String
        get() = ""
}

data class ImmutableSpan(
    override val component: String,
    override val name: String,
    override val startTime: Long,
    override val endTime: Long,
    override val tags: Map<String, String> = mapOf(),
    override val errorCode: String = ErrorCodes.SUCCEEDED,
    override val errorMsg: String = ErrorCodes.SUCCEEDED_MSG,
    override val stackTrace: String = ""
) : Span

data class MutableSpan(
    override val component: String,
    override val name: String,
    override val startTime: Long,
    override var endTime: Long = Long.MIN_VALUE,
    override val tags: MutableMap<String, String> = mutableMapOf(),
    override var errorCode: String = ErrorCodes.SUCCEEDED,
    override var errorMsg: String = ErrorCodes.SUCCEEDED_MSG,
    override var stackTrace: String = ""
) : Span {

    fun success() {
        endTime = Clock.systemUTC().millis()
    }

    fun error(throwable: Throwable) {
        endTime = Clock.systemUTC().millis()
        val errorInfo = throwable.asErrorInfo()
        errorCode = errorInfo.errorCode
        errorMsg = errorInfo.errorMsg
        stackTrace = throwable.stackTraceToString()
    }
}

data class SimpleTracingMessage(
    override val contextName: String,
    override val id: String,
    override val header: Header,
    override val body: Span,
    override val createTime: Long = Clock.systemUTC().millis()
) : TracingMessage {
    override fun mergeHeader(additionalSource: Map<String, String>): TracingMessage {
        return copy(header = header.mergeWith(additionalSource))
    }
}

fun Span.asTracingMessage(
    contextName: String,
    id: String = IdFactory.generateAsString(),
    header: Header = Header.EMPTY,
    createTime: Long = Clock.systemUTC().millis()
): TracingMessage {
    return SimpleTracingMessage(
        contextName = contextName,
        id = id,
        header = header,
        body = this,
        createTime = createTime
    )
}
