/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.modeling

import me.ahoo.wow.Identifier
import me.ahoo.wow.id.generateId

internal fun AggregateId.equalTo(other: Any?): Boolean {
    if (this === other) return true
    if (other !is AggregateId) return false

    if (contextName != other.contextName) return false
    if (aggregateName != other.aggregateName) return false
    if (id != other.id) return false

    return true
}

private const val HASH_MAGIC = 31
internal fun AggregateId.hash(): Int {
    var result = contextName.hashCode()
    result = HASH_MAGIC * result + aggregateName.hashCode()
    result = HASH_MAGIC * result + id.hashCode()
    return result
}

/**
 * Aggregate Id .
 *
 * @author ahoo wang
 */
interface AggregateId : Identifier, NamedAggregate, NamedAggregateDecorator, Comparable<AggregateId> {
    /**
     * @see MaterializedNamedAggregate
     */
    override val namedAggregate: NamedAggregate

    override fun compareTo(other: AggregateId): Int {
        require(isSameAggregateName(other)) {
            "NamedAggregate[$namedAggregate VS ${other.namedAggregate}] are different and cannot be compared."
        }
        return id.compareTo(other.id)
    }
}

interface TypedAggregateId<A : Any> : AggregateId, TypedAggregate<A>

data class DefaultAggregateId(
    override val namedAggregate: NamedAggregate,
    override val id: String
) : AggregateId {
    override fun equals(other: Any?): Boolean = equalTo(other)

    override fun hashCode(): Int = hash()
}

data class DefaultTypedAggregateId<A : Any>(
    override val namedAggregate: NamedAggregate,
    override val aggregateType: Class<A>,
    override val id: String
) : TypedAggregateId<A> {
    override fun equals(other: Any?): Boolean = equalTo(other)
    override fun hashCode(): Int = hash()
}

fun NamedAggregate.asAggregateId(id: String = generateId()) = DefaultAggregateId(
    namedAggregate = materialize(),
    id = id
)

fun <A : Any> NamedTypedAggregate<A>.asTypedAggregateId(
    id: String = generateId()
) = DefaultTypedAggregateId(
    namedAggregate = materialize(),
    aggregateType = aggregateType,
    id = id
)
