/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.modeling.command

import me.ahoo.wow.command.ServerCommandExchange
import me.ahoo.wow.event.DomainEventBus
import me.ahoo.wow.messaging.function.logErrorResume
import me.ahoo.wow.messaging.handler.Filter
import me.ahoo.wow.messaging.handler.FilterChain
import me.ahoo.wow.messaging.handler.FilterType
import reactor.core.publisher.Mono
import reactor.core.scheduler.Schedulers

@FilterType(AggregateDispatcher::class)
class AggregateProcessorFilter(
    private val domainEventBus: DomainEventBus
) : Filter<ServerCommandExchange<Any>> {
    override fun filter(
        exchange: ServerCommandExchange<Any>,
        chain: FilterChain<ServerCommandExchange<Any>>
    ): Mono<Void> {
        return Mono.deferContextual { context ->
            val aggregateProcessor = context.get(AggregateProcessor::class.java)
            aggregateProcessor.process(exchange.message)
                .doFinally {
                    exchange.acknowledge()
                }
                // 使用单独线程，保证事件有序的发送到 DomainEventBus
                .publishOn(Schedulers.single())
                .flatMap {
                    domainEventBus
                        .send(it)
                        .logErrorResume()
                }
        }
    }
}
