/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.ahoo.wow.modeling.command

import me.ahoo.wow.command.CommandMessage
import me.ahoo.wow.event.DomainEventStream
import me.ahoo.wow.event.asDomainEventStream
import me.ahoo.wow.eventsourcing.EventStore
import me.ahoo.wow.ioc.ServiceProvider
import me.ahoo.wow.messaging.asHeader
import me.ahoo.wow.messaging.tracing.MessageTracer
import me.ahoo.wow.messaging.tracing.getUpstream
import me.ahoo.wow.modeling.NamedTypedAggregate
import me.ahoo.wow.modeling.matedata.CommandAggregateMetadata
import me.ahoo.wow.modeling.state.StateAggregate
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import reactor.core.publisher.Mono
import reactor.kotlin.core.publisher.toMono
import java.lang.IllegalArgumentException

class SimpleCommandAggregate<C : Any, S : Any>(
    override val state: StateAggregate<S>,
    override val commandRoot: C,
    private val eventStore: EventStore,
    private val metadata: CommandAggregateMetadata<C>,
    private val serviceProvider: ServiceProvider
) : CommandAggregate<C, S>,
    NamedTypedAggregate<C> by metadata {
    private companion object {
        private val log: Logger = LoggerFactory.getLogger(SimpleCommandAggregate::class.java)
    }

    private val functionRegistry = metadata.asMessageFunctionRegistry(commandRoot)

    override fun process(message: CommandMessage<*>): Mono<DomainEventStream> {
        return Mono.deferContextual { context ->
            if (log.isDebugEnabled) {
                log.debug("Process {}.", message)
            }
            if (message.aggregateVersion != null &&
                message.aggregateVersion != version
            ) {
                return@deferContextual IncompatibleVersionException(
                    command = message,
                    expectVersion = message.aggregateVersion!!,
                    actualVersion = version
                ).toMono()
            }

            if (!message.isCreate && !initialized) {
                return@deferContextual IllegalArgumentException(
                    "Illegal execution of the $message, the current aggregation state is not initialized."
                ).toMono()
            }

            val commandFunction = functionRegistry[message.body.javaClass]
            requireNotNull(commandFunction) {
                "CommandType[${message.body.javaClass}]'s OnCommandMetadata not found."
            }
            val tracedHeader = MessageTracer.DEFAULT
                .trace(context.getUpstream() ?: message)
                .asHeader()
            commandFunction
                .handle(message, serviceProvider)
                .map {
                    it.asDomainEventStream(
                        command = message,
                        aggregateVersion = version,
                        header = tracedHeader
                    )
                }
                .doOnNext {
                    /**
                     * 将领域事件朔源到当前状态聚合根.
                     */
                    commandState = commandState.onSourcing(state, it)
                }
                .flatMap { eventStream ->
                    /**
                     * 持久化事件存储,完成持久化领域事件意味着 命令已经完成.
                     */
                    commandState.onStore(eventStore, eventStream)
                        .doOnNext { commandState = it }
                        .doOnError { commandState = CommandState.EXPIRED }
                        .thenReturn(eventStream)
                }
        }
    }

    override var commandState = CommandState.STORED

    override fun toString(): String {
        return "SimpleCommandAggregate(state=$state, metadata=$metadata, commandState=$commandState)"
    }
}
