/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.ahoo.wow.query

import me.ahoo.wow.naming.NamedBoundedContext
import org.slf4j.LoggerFactory
import reactor.core.publisher.Flux
import reactor.core.publisher.Mono
import reactor.core.publisher.Sinks

class InMemoryQueryBus(
    private val sink: Sinks.Many<QueryExchange<Any, Any>> = Sinks.many().multicast().onBackpressureBuffer()
) : QueryBus {
    companion object {
        private val log = LoggerFactory.getLogger(InMemoryQueryBus::class.java)
    }

    override fun <Q : Any, R : Any> send(queryMessage: QueryMessage<Q>): Mono<R> {
        return Mono.defer {
            if (log.isDebugEnabled) {
                log.debug("Send {}.", queryMessage)
            }
            val queryExchange = DefaultQueryExchange<Q, R>(queryMessage)
            @Suppress("UNCHECKED_CAST")
            sink.emitNext(queryExchange as QueryExchange<Any, Any>, Sinks.EmitFailureHandler.FAIL_FAST)
            queryExchange.result
        }
    }

    override fun receive(namedBoundedContexts: Set<NamedBoundedContext>): Flux<QueryExchange<Any, Any>> {
        return sink.asFlux()
            .filter { queryExchange ->
                namedBoundedContexts.any {
                    it.isSameBoundedContext(queryExchange.message)
                }
            }
    }
}
