/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.query

import me.ahoo.wow.messaging.tracing.getUpstream
import me.ahoo.wow.messaging.tracing.trace
import me.ahoo.wow.naming.NamedBoundedContext
import reactor.core.publisher.Flux
import reactor.core.publisher.Mono

interface QueryGateway : QueryBus {
    fun <Q : Any, R : Any> send(query: Q): Mono<R>
}

class DefaultQueryGateway(
    private val queryBus: QueryBus
) : QueryGateway {
    override fun <Q : Any, R : Any> send(query: Q): Mono<R> {
        return Mono.deferContextual {
            val queryMessage = query.asQueryMessage().trace(it.getUpstream())
            return@deferContextual queryBus.send<Q, R>(queryMessage)
        }
    }

    override fun <Q : Any, R : Any> send(queryMessage: QueryMessage<Q>): Mono<R> {
        return Mono.deferContextual {
            val tracedQuery = queryMessage.trace(it.getUpstream())
            return@deferContextual queryBus.send<Q, R>(tracedQuery)
        }
    }

    override fun receive(namedBoundedContexts: Set<NamedBoundedContext>): Flux<QueryExchange<Any, Any>> {
        return queryBus.receive(namedBoundedContexts)
    }
}
