/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.command

import me.ahoo.wow.api.Version
import me.ahoo.wow.api.command.CommandMessage
import me.ahoo.wow.api.messaging.Header
import me.ahoo.wow.api.modeling.NamedAggregate
import me.ahoo.wow.api.modeling.TenantId
import me.ahoo.wow.command.annotation.commandMetadata
import me.ahoo.wow.id.GlobalIdGenerator
import me.ahoo.wow.id.generateId
import me.ahoo.wow.messaging.DefaultHeader
import me.ahoo.wow.modeling.aggregateId

@Suppress("LongParameterList")
fun <C : Any> C.toCommandMessage(
    id: String = GlobalIdGenerator.generateAsString(),
    requestId: String? = null,
    aggregateId: String? = null,
    tenantId: String? = null,
    aggregateVersion: Int? = null,
    namedAggregate: NamedAggregate? = null,
    header: Header = DefaultHeader.empty(),
    createTime: Long = System.currentTimeMillis()
): CommandMessage<C> {
    val metadata = javaClass.commandMetadata()
    val commandNamedAggregate = namedAggregate ?: metadata.namedAggregateGetter?.getNamedAggregate(this)
    requireNotNull(commandNamedAggregate) {
        "The command[$javaClass] must be associated with a named aggregate!"
    }
    val commandAggregateId = metadata.aggregateIdGetter?.get(this) ?: aggregateId ?: commandNamedAggregate.generateId()
    val commandTenantId = metadata.tenantIdGetter?.get(this) ?: tenantId ?: TenantId.DEFAULT_TENANT_ID
    val targetAggregateId = commandNamedAggregate.aggregateId(id = commandAggregateId, tenantId = commandTenantId)
    val expectedAggregateVersion = if (metadata.isCreate) {
        Version.UNINITIALIZED_VERSION
    } else {
        metadata.aggregateVersionGetter?.get(this) ?: aggregateVersion
    }

    return SimpleCommandMessage(
        id = id,
        requestId = requestId ?: id,
        header = header,
        body = this,
        createTime = createTime,
        aggregateId = targetAggregateId,
        aggregateVersion = expectedAggregateVersion,
        name = metadata.name,
        isCreate = metadata.isCreate,
        allowCreate = metadata.allowCreate,
    )
}

@Deprecated(
    "Please use toCommandMessage instead.",
    replaceWith = ReplaceWith(
        "toCommandMessage(id, requestId, aggregateId, tenantId, aggregateVersion, namedAggregate, header, createTime)"
    )
)
@Suppress("LongParameterList")
fun <C : Any> C.asCommandMessage(
    id: String = GlobalIdGenerator.generateAsString(),
    requestId: String? = null,
    aggregateId: String? = null,
    tenantId: String? = null,
    aggregateVersion: Int? = null,
    namedAggregate: NamedAggregate? = null,
    header: Header = DefaultHeader.empty(),
    createTime: Long = System.currentTimeMillis()
): CommandMessage<C> {
    return toCommandMessage(
        id,
        requestId,
        aggregateId,
        tenantId,
        aggregateVersion,
        namedAggregate,
        header,
        createTime
    )
}
