/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.modeling.state

import me.ahoo.wow.api.Version
import me.ahoo.wow.api.modeling.AggregateId
import me.ahoo.wow.configuration.requiredAggregateType
import me.ahoo.wow.modeling.aggregateId
import me.ahoo.wow.modeling.annotation.aggregateMetadata
import me.ahoo.wow.modeling.matedata.AggregateMetadata
import me.ahoo.wow.modeling.matedata.StateAggregateMetadata
import org.slf4j.Logger
import org.slf4j.LoggerFactory
import reactor.core.publisher.Mono

/**
 * Aggregate Factory .
 * 创建一个未初始化/空状态的聚合.
 *
 * @author ahoo wang
 */
interface StateAggregateFactory {
    fun <S : Any> create(metadata: StateAggregateMetadata<S>, aggregateId: AggregateId): StateAggregate<S>
    fun <S : Any> createAsMono(metadata: StateAggregateMetadata<S>, aggregateId: AggregateId): Mono<StateAggregate<S>> {
        return Mono.fromCallable {
            create(metadata, aggregateId)
        }
    }
}

object ConstructorStateAggregateFactory : StateAggregateFactory {
    private val log: Logger = LoggerFactory.getLogger(ConstructorStateAggregateFactory::class.java)

    override fun <S : Any> create(
        metadata: StateAggregateMetadata<S>,
        aggregateId: AggregateId
    ): StateAggregate<S> {
        val stateRoot = metadata.constructState(aggregateId)
        return create(
            metadata = metadata,
            aggregateId = aggregateId,
            state = stateRoot,
            version = Version.UNINITIALIZED_VERSION
        )
    }

    fun <S : Any> create(
        metadata: StateAggregateMetadata<S>,
        aggregateId: AggregateId,
        state: S,
        version: Int,
        eventId: String = "",
        firstOperator: String = "",
        operator: String = "",
        firstEventTime: Long = 0,
        eventTime: Long = 0,
        deleted: Boolean = false
    ): StateAggregate<S> {
        if (log.isDebugEnabled) {
            log.debug("Create {}.", aggregateId)
        }
        val stateAggregate = SimpleStateAggregate(
            aggregateId = aggregateId,
            metadata = metadata,
            state = state,
            version = version,
            eventId = eventId,
            firstOperator = firstOperator,
            operator = operator,
            firstEventTime = firstEventTime,
            eventTime = eventTime,
            deleted = deleted,
        )
        if (state is ReadOnlyStateAggregateAware<*>) {
            @Suppress("UNCHECKED_CAST")
            val aware = state as ReadOnlyStateAggregateAware<S>
            aware.setReadOnlyStateAggregate(stateAggregate)
        }
        return stateAggregate
    }

    private fun <S : Any> StateAggregateMetadata<S>.constructState(aggregateId: AggregateId): S {
        if (constructorAccessor.constructor.parameterCount == 1) {
            return constructorAccessor.invoke(arrayOf(aggregateId.id))
        }
        return constructorAccessor.invoke(arrayOf(aggregateId.id, aggregateId.tenantId))
    }

    @JvmStatic
    fun <S : Any> StateAggregateMetadata<S>.toStateAggregate(
        aggregateId: AggregateId,
        state: S,
        version: Int,
        eventId: String = "",
        firstOperator: String = "",
        operator: String = "",
        firstEventTime: Long = 0,
        eventTime: Long = 0,
        deleted: Boolean = false
    ): StateAggregate<S> {
        return create(
            metadata = this,
            aggregateId = aggregateId,
            state = state,
            version = version,
            eventId = eventId,
            firstOperator = firstOperator,
            operator = operator,
            firstEventTime = firstEventTime,
            eventTime = eventTime,
            deleted = deleted
        )
    }

    @JvmStatic
    fun <S : Any> AggregateMetadata<*, S>.toStateAggregate(
        state: S,
        version: Int,
        eventId: String = "",
        firstOperator: String = "",
        operator: String = "",
        firstEventTime: Long = 0,
        eventTime: Long = 0,
        deleted: Boolean = false
    ): StateAggregate<S> {
        val aggregateId = aggregateId(this.state.aggregateIdAccessor[state])
        return this.state.toStateAggregate(
            aggregateId = aggregateId,
            state = state,
            version = version,
            eventId = eventId,
            firstOperator = firstOperator,
            operator = operator,
            firstEventTime = firstEventTime,
            eventTime = eventTime,
            deleted = deleted,
        )
    }

    @JvmStatic
    fun <S : Any> ReadOnlyStateAggregate<S>.toStateAggregate(): StateAggregate<S> {
        val metadata = aggregateId.requiredAggregateType<Any>()
            .aggregateMetadata<Any, S>().state
        return metadata.toStateAggregate(
            aggregateId = aggregateId,
            state = state,
            version = version,
            eventId = eventId,
            firstOperator = firstOperator,
            operator = operator,
            firstEventTime = firstEventTime,
            eventTime = eventTime,
            deleted = deleted,
        )
    }
}
