/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.command.wait

import io.github.oshai.kotlinlogging.KotlinLogging
import java.util.concurrent.ConcurrentHashMap

/**
 * 命令结果注册器.
 */
interface WaitStrategyRegistrar {
    /**
     * 注册.
     *
     * @see java.util.Map.putIfAbsent
     */
    fun register(waitStrategy: WaitStrategy): WaitStrategy?

    /**
     * 取消注册.
     *
     * @see java.util.Map.remove
     */
    fun unregister(waitCommandId: String): WaitStrategy?

    fun get(waitCommandId: String): WaitStrategy?

    operator fun contains(waitCommandId: String): Boolean

    fun next(signal: WaitSignal): Boolean {
        val waitStrategy = get(signal.waitCommandId) ?: return false
        waitStrategy.next(signal)
        return true
    }
}

object SimpleWaitStrategyRegistrar : WaitStrategyRegistrar {
    private val log = KotlinLogging.logger {}
    private val waitStrategies: ConcurrentHashMap<String, WaitStrategy> = ConcurrentHashMap()

    override fun register(waitStrategy: WaitStrategy): WaitStrategy? {
        log.debug {
            "Register - waitCommandId[${waitStrategy.waitCommandId}] WaitStrategy."
        }
        return waitStrategies.putIfAbsent(waitStrategy.waitCommandId, waitStrategy)
    }

    override fun unregister(waitCommandId: String): WaitStrategy? {
        val value = waitStrategies.remove(waitCommandId)
        log.debug {
            "Unregister - remove waitCommandId[$waitCommandId] WaitStrategy - [${value != null}]."
        }
        return value
    }

    override fun get(waitCommandId: String): WaitStrategy? {
        return waitStrategies[waitCommandId]
    }

    override fun contains(waitCommandId: String): Boolean {
        return waitStrategies.containsKey(waitCommandId)
    }
}
