/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.ahoo.wow.domains.order.saga

import me.ahoo.cosid.IdGenerator
import me.ahoo.wow.domains.order.command.ShipOrder
import me.ahoo.wow.domains.order.event.OrderCancelled
import me.ahoo.wow.domains.order.event.OrderCreated
import me.ahoo.wow.domains.order.event.OrderPaid
import me.ahoo.wow.domains.order.event.OrderShipped
import me.ahoo.wow.domains.order.model.PayOrder
import me.ahoo.wow.domains.payment.command.PreparePayment
import me.ahoo.wow.domains.payment.command.Refund
import me.ahoo.wow.domains.payment.event.PaymentPaid
import me.ahoo.wow.domains.warehouse.command.PrepareShip
import me.ahoo.wow.domains.warehouse.command.Ship
import me.ahoo.wow.domains.warehouse.event.Shipped
import me.ahoo.wow.saga.annotation.Saga

@Saga
class OrderProcessManager(private val idGenerator: IdGenerator) {
    private var processState: ProcessState? = null

    /**
     * start saga.
     * <pre>
     * 1. Saga.Id=启动事件聚合ID o-- orderId/paymentId/shipId
     * 2. 发送准备支付命令到支付服务
     * 3. 可产生多条命令
     * 4. 根据命令聚合ID生成Saga关联事件(内置事件)，持久化到 EventStore.
     * -----
     * 通过 onEvent 朔源
     </pre> *
     */
    fun onEvent(orderCreated: OrderCreated): Iterable<*> {
        processState = ProcessState.CREATED
        val preparePayment = PreparePayment(idGenerator.generateAsString(), orderCreated.orderId)
        val prepareShip = PrepareShip(idGenerator.generateAsString(), orderCreated.orderId)
        return listOf(preparePayment, prepareShip)
    }

    fun onEvent(paymentPaid: PaymentPaid): PayOrder {
        processState = ProcessState.PAYMENT_PAID
        return PayOrder(paymentPaid.bizId, paymentPaid.paymentId, paymentPaid.amount)
    }

    fun onEvent(orderPaid: OrderPaid): Ship {
        processState = ProcessState.ORDER_PAID
        return Ship(orderPaid.orderId)
    }

    fun onEvent(shipped: Shipped): ShipOrder {
        processState = ProcessState.SHIPPED
        return ShipOrder(shipped.bizId)
    }

    fun onEvent(orderCancelled: OrderCancelled?): Iterable<*> {
        val preState = processState
        processState = ProcessState.ORDER_CANCELLED
        return when (processState) {
            ProcessState.CREATED -> {
                listOf<Any>()
            }
            ProcessState.PAYMENT_PAID -> {
                // 退款命令 and 取消发货命令
                listOf(Refund())
            }
            else -> throw IllegalStateException("Unexpected value: $processState")
        }
    }

    /**
     * end saga.
     */
    fun onEvent(orderShipped: OrderShipped?) {
        processState = ProcessState.ORDER_SHIPPED
    }

    internal enum class ProcessState {
        CREATED, PAYMENT_PAID, ORDER_PAID, INVENTORY_LOCKED, SHIPPED, ORDER_SHIPPED, ORDER_CANCELLED
    }
}
