/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.query.filter

import me.ahoo.wow.api.modeling.NamedAggregate
import me.ahoo.wow.api.query.Condition
import me.ahoo.wow.api.query.IListQuery
import me.ahoo.wow.api.query.IPagedQuery
import me.ahoo.wow.api.query.ISingleQuery
import me.ahoo.wow.api.query.MaterializedSnapshot
import me.ahoo.wow.api.query.PagedList
import reactor.core.publisher.Flux
import reactor.core.publisher.Mono
import java.util.concurrent.ConcurrentHashMap

const val QUERY_KEY = "__QUERY__"
const val RESULT_KEY = "__RESULT__"

@Suppress("UNCHECKED_CAST")
interface SnapshotQueryContext<SOURCE : SnapshotQueryContext<SOURCE, Q, R>, Q : Any, R : Any> {
    val attributes: MutableMap<String, Any>
    val namedAggregate: NamedAggregate
    val queryType: QueryType

    fun setQuery(query: Q): SOURCE {
        attributes[QUERY_KEY] = query
        return this as SOURCE
    }

    fun getQuery(): Q {
        return checkNotNull(attributes[QUERY_KEY]) as Q
    }

    fun setResult(result: R): SOURCE {
        attributes[RESULT_KEY] = result
        return this as SOURCE
    }

    fun getRequiredResult(): R {
        return checkNotNull(attributes[RESULT_KEY]) as R
    }
}

enum class QueryType {
    SINGLE,
    LIST,
    PAGED,
    COUNT
}

class SingleSnapshotQueryContext<S : Any>(
    override val namedAggregate: NamedAggregate,
    override val attributes: MutableMap<String, Any> = ConcurrentHashMap(),
) : SnapshotQueryContext<SingleSnapshotQueryContext<S>, ISingleQuery, Mono<MaterializedSnapshot<S>>> {
    override val queryType: QueryType
        get() = QueryType.SINGLE
}

class ListSnapshotQueryContext<S : Any>(
    override val namedAggregate: NamedAggregate,
    override val attributes: MutableMap<String, Any> = ConcurrentHashMap(),
) : SnapshotQueryContext<ListSnapshotQueryContext<S>, IListQuery, Flux<MaterializedSnapshot<S>>> {
    override val queryType: QueryType
        get() = QueryType.LIST
}

class PagedSnapshotQueryContext<S : Any>(
    override val namedAggregate: NamedAggregate,
    override val attributes: MutableMap<String, Any> = ConcurrentHashMap(),
) : SnapshotQueryContext<PagedSnapshotQueryContext<S>, IPagedQuery, Mono<PagedList<MaterializedSnapshot<S>>>> {
    override val queryType: QueryType
        get() = QueryType.PAGED
}

class CountSnapshotQueryContext(
    override val namedAggregate: NamedAggregate,
    override val attributes: MutableMap<String, Any> = ConcurrentHashMap(),
) : SnapshotQueryContext<CountSnapshotQueryContext, Condition, Mono<Long>> {
    override val queryType: QueryType
        get() = QueryType.COUNT
}
