/*
 * Copyright [2021-present] [ahoo wang <ahoowang@qq.com> (https://github.com/Ahoo-Wang)].
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *      http://www.apache.org/licenses/LICENSE-2.0
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package me.ahoo.wow.spring

import org.slf4j.LoggerFactory
import org.springframework.context.ApplicationContext
import org.springframework.context.SmartLifecycle
import org.springframework.context.SmartLifecycle.DEFAULT_PHASE
import java.util.concurrent.atomic.AtomicBoolean

/**
 * must before Launcher
 * @see MessageDispatcherLauncher
 */
const val AUTO_REGISTRAR_PHASE = DEFAULT_PHASE - 100

abstract class AutoRegistrar<CM : Annotation>(
    private val componentType: Class<CM>,
    private val applicationContext: ApplicationContext,
) : SmartLifecycle {
    companion object {
        private val log = LoggerFactory.getLogger(AutoRegistrar::class.java)
    }

    private val running = AtomicBoolean(false)

    override fun start() {
        if (log.isInfoEnabled) {
            log.info("Start registering component:${componentType.simpleName}.")
        }
        if (!running.compareAndSet(false, true)) {
            return
        }
        val components = applicationContext.getBeansWithAnnotation(componentType)
        components.forEach { entry ->
            val component = entry.value
            if (log.isDebugEnabled) {
                log.debug("Registering Component {}.", component)
            }
            register(component)
        }
    }

    abstract fun register(component: Any)

    override fun stop() {
        if (log.isInfoEnabled) {
            log.info("Stop ${componentType.simpleName}.")
        }
        if (!running.compareAndSet(true, false)) {
            return
        }
    }

    override fun isRunning(): Boolean {
        return running.get()
    }

    override fun getPhase(): Int {
        return AUTO_REGISTRAR_PHASE
    }
}
