package me.chatgame.voip;

import android.app.Activity;
import android.content.Context;
import android.graphics.Bitmap;
import android.hardware.SensorManager;
import android.media.AudioManager;
import android.net.Uri;
import android.os.Environment;
import android.os.Handler;
import android.text.TextUtils;
import android.view.OrientationEventListener;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.locks.ReentrantLock;

import me.chatgame.voip.VoipAudioIO.AUDIO_IO_TYPE;
import me.chatgame.voip.VoipAudioIO.ClearAudioCacheCallback;
import me.chatgame.voip.VoipVideoCapture.VideoCaptureCallback;

/**
 * Voip object to provide all voip functions.
 */
public class VoipAndroid implements ClearAudioCacheCallback,
		VideoCaptureCallback {

	public enum MP3_TYPE{
		RINGTONE,
		NOTIFICATION,
		MUSIC
	}

	public enum WindowSizeLevel {
		SMALL(0),
		MEDIUM(1),
		LARGE(2);

		private int index;

		private WindowSizeLevel(int index) {
			this.index = index;
		}
	}

	/**
	 * CPU Level equals to iPhone4
	 */
	public static final int CPU_IPHONE4 = 0;

	/**
	 * CPU Level equals to iPhone4S
	 */
	public static final int CPU_IPHONE4S = 1;

	/**
	 * CPU Level equals to iPhone5
	 */
	public static final int CPU_IPHONE5 = 2;

	/**
	 * CPU Level equals to iPhone5S
	 */
	public static final int CPU_IPHONE5S = 3;

	/**
	 * CPU Level equals to iPhone6
	 */
	public static final int CPU_IPHONE6 = 4;

	/**
	 * VOIP compatible mode which can make audio/video call with original
	 * version.
	 */
	public static final int VOIP_MODE_COMPATIBLE = 0;

	/**
	 * VOIP udp mode which use UDP protocol to transfer audio/video data
	 * directly.
	 */
	public static final int VOIP_MODE_UDP = 1;

	/**
	 * VOIP udp+ mode which use UDP+ to transfer audio/video data.
	 */
	public static final int VOIP_MODE_UDPLUS = 2;

	/**
	 * Do NOT use face beauty.
	 */
	public static final int FACEBEAUTY_NONE = 0;

	/**
	 * Classic face beauty mode.
	 */
	public static final int FACEBEAUTY_CLASSIC = 1;

	/**
	 * Warm yellow face beauty mode.
	 */
	public static final int FACEBEAUTY_WARM_YELLOW = 2;

	/**
	 * Simple elegant face beauty mode.
	 */
	public static final int FACEBEAUTY_SIMPLE_ELEGANT = 3;

	/**
	 * Cold cool face beauty mode.
	 */
	public static final int FACEBEAUTY_COLD_COOL = 4;

	/**
	 * Black white face beauty mode.
	 */
	public static final int FACEBEAUTY_BLACK_WHITE = 5;

	/**
	 * VOIP log level DISABLE
	 */
	public static final int LOG_DISABLE = -1;

	/**
	 * VOIP log level ERROR
	 */
	public static final int LOG_ERROR = 0;

	/**
	 * VOIP log level WARNING
	 */
	public static final int LOG_WARNING = 1;

	/**
	 * VOIP log level INFO
	 */
	public static final int LOG_INFO = 2;

	/**
	 * VOIP log level DEBUG
	 */
	public static final int LOG_DEBUG = 3;

	/**
	 * Audio parameter class.
	 */
	public static class AudioParameter {
		/**
		 * Audio bitrate.
		 */
		public int bitrate;

		/**
		 * Indicates whether audio is used for game. The AEC will be different
		 * on chat and game.
		 */
		public boolean isGameMode;

		/**
		 * Indicates only receiving audio without sending
		 */
		public boolean recvOnly;

		/**
		 * Constructor.
		 *
		 * @param bitrate
		 * @param isGameMode
		 * @param recvOnly
		 */
		public AudioParameter(int bitrate, boolean isGameMode, boolean recvOnly) {
			this.bitrate = bitrate;
			this.isGameMode = isGameMode;
			this.recvOnly = recvOnly;
		}
	}

	/**
	 * Video parameter class.
	 */
	public static class VideoParameter {
		/**
		 * Video initialized bitrate. Video bitrate may be changed during a call
		 * because of DBA.
		 */
		public int bitrate;

		/**
		 * Indicates only receiving video without sending
		 */
		public boolean recvOnly;

		/**
		 * Constructor.
		 *
		 * @param bitrate
		 * @param recvOnly
		 */
		public VideoParameter(int bitrate, boolean recvOnly) {
			this.bitrate = bitrate;
			this.recvOnly = recvOnly;
		}
	}

	/**
	 * Live Video Parameter class.
	 */
	public static class LiveVideoParameter {
		/**
		 * Composition: Positions of front video will start from bottom to top or from top to bottom
		 */
		public boolean comp_bottomToTop;

		/**
		 * Composition: Positions of front video will start from right to left or from left to right
		 */
		public boolean comp_rightToLeft;

		/**
		 * Composition: scale proportion of front video to back video
		 */
		public float comp_scaleProportion;

		/**
		 * Composition: vertical margin, proportion to back video's heigth
		 */
		public float comp_marginVer;

		/**
		 * Composition: horizontal margin, proportion to back video's width
		 */
		public float comp_marginHor;

		/**
		 * Max video bitrate for live host. If 0, no restrict.
		 */
		public int host_maxBitrate;

		/**
		 * Max video bitrate for live guest. If 0, no restrict.
		 */
		public int guest_maxBitrate;

		/**
		 * Constructor.
		 *
		 * @param comp_bToT
		 * @param comp_rToL
		 * @param comp_scale
		 * @param comp_mgVer
		 * @param comp_mgHor
		 * @param host_maxBr
		 * @param guest_maxBr
		 */
		public LiveVideoParameter(boolean comp_bToT, boolean comp_rToL, float comp_scale, float comp_mgVer, float comp_mgHor, int host_maxBr, int guest_maxBr) {
			this.comp_bottomToTop = comp_bToT;
			this.comp_rightToLeft = comp_rToL;
			this.comp_scaleProportion = comp_scale;
			this.comp_marginVer = comp_mgVer;
			this.comp_marginHor = comp_mgHor;
			this.host_maxBitrate = host_maxBr;
			this.guest_maxBitrate = guest_maxBr;
		}
	}

	private boolean inited = false;
	private long context = 0;
	private long contextCallback;
	private Handler handler;

	private static boolean loadSdcardLibrary = false;
	private static boolean loadSdcardAudioConfig = false;

	private VoipAudioIO audioIO;
	private VoipAudioEffect audioEffect;
	private VoipVideoCapture videoCapture;
	private int deviceOrientation = 0;
	private boolean inAudioCall = false;
	private ReentrantLock inAudioCallLock = new ReentrantLock();
	AudioManager audioManager = null;

	private static boolean hasSetUserAgent = false;

	public interface AudioCallback {
		public void audioIsWind(boolean isWind);

		/*
		 * @param dB, range: 0 ~ 60
		 */
		public void audioDB(int dB);

		/*
		 * @param code -1: request audio focus failure -2: device initialization
		 * failure -3: audio focus loss -4: all-zero PCM data recorded
		 */
		public void audioNotification(int code);

		public void remoteAudioChanged(boolean enable);
	}

	private AudioCallback audioCallback = null;

	public interface VideoCallback {
		public void previewImage(VoipImage image);

		public void decodedImage(VoipImage image);

		public void remoteVideoChanged(boolean enable);

		public void localVideoChanged(boolean enable);
	}

	private VideoCallback videoCallback = null;

	private class ThreadCallback extends Thread {
		public Boolean toStop = false;

		public ThreadCallback() {
			this.toStop = false;
		}

		public void run() {
			int index = 0;
			while (!toStop) {
				if (index == 3 && audioCallback != null) {
					audioCallback.audioIsWind(audioIsWind());
					audioCallback.audioDB(audioDB());
					index = 0;
				}

				try {
					sleep(15);
					index = index + 1;
				} catch (InterruptedException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
		}
	}

	private ThreadCallback threadCallback;

	public interface StreamMediaPlayerCallback {
		/*
		 * @param mediaUrl
		 * @param code -1 : stream open failed
		 */
		public void streamMediaPlayerNotify(String mediaUrl, int code);

		/*
		 * @param image decoded video frame
		 */
		public void streamMediaDecodedImage(VoipImage image);
	}

	public interface VideoMsgRecorderCallback {
		/*
		 * @param code -1: wrong status -2: file open failed -3: file write
		 * failed -4: message too long -5: internal error
		 */
		public void videoMsgRecorderNotify(int code);

		/*
		 * It's repeatedly called during video message recording. If
		 * "isFileTailer" is true, "data" will be the file tailer.
		 * 
		 * @param data video message file content
		 * 
		 * @param len valid data length
		 * 
		 * @param isFileTailer "true" means recording ended
		 */
		public void videoMsgData(byte[] data, int len, boolean isFileTailer);

		/*
		 * @param image the first video frame
		 */
		public void videoMsgFirstFrame(VoipImage image);
	}

	public interface VideoMsgPlayerCallback {
		/*
		 * @param fileName
		 * @param code 0 : play completed -1: file open failed -2: file format
		 * error
		 */
		public void videoMsgPlayerNotify(String fileName, int code);

		/*
		 * @param image decoded video frame
		 */
		public void videoMsgDecodedImage(VoipImage image);
	}

	public interface AudioMsgRecorderCallback {
		/*
		 * @param code -1: wrong status -2: file open failed -3: file write
		 * failed -4: message too long -5: internal error
		 */
		public void audioMsgRecorderNotify(int code);

		/*
		 * It's repeatedly called during audio message recording. If
		 * "isFileHeader" is true, "data" will be the file header.
		 * 
		 * @param data audio message file content
		 * 
		 * @param len valid data length
		 * 
		 * @param isFileHeader "true" means recording ended
		 */
		public void audioMsgData(byte[] data, int len, boolean isFileHeader);
	}

	public interface AudioMsgPlayerCallback {
		/*
		 * @param code 0 : play completed -1: file open failed -2: file format
		 * error
		 */
		public void audioMsgPlayerNotify(int code);
	}

	private StreamMediaPlayerCallback streamMediaPlayerCallback = null;
	private boolean videoMsgAllStopped = false;
	private Map<String, List<VideoMsgPlayerCallback>> videoMsgPlayerCallbacks = new HashMap<String, List<VideoMsgPlayerCallback>>();
	private ReentrantLock videoMsgPlayerCallbacksLock = new ReentrantLock();
	private VideoMsgRecorderCallback videoMsgRecorderCallback = null;
	private AudioMsgRecorderCallback audioMsgRecorderCallback = null;
	private AudioMsgPlayerCallback audioMsgPlayerCallback = null;
	private String audioMsgFileName = null;


	private VoipConfig config = null;

	public VoipConfig getConfig() {
		return this.config;
	}

	private OrientationEventListener orientationListener;

	/**
	 *
	 * @param handler
	 * @param context
	 * @param jsonConfig
	 */
	public VoipAndroid(Handler handler, Context context, JSONObject jsonConfig) {

		if(loadSdcardLibrary) {
			this.loadVoipLibraryfromSDCard(context);
		}

		String manufacturer = android.os.Build.MANUFACTURER;
		String device = android.os.Build.DEVICE;
		String model = android.os.Build.MODEL;
		VoipLog.i("[VoipAndroid] manufactruer=" + manufacturer + ", device="
				+ device + ", model=" + model);

		this.handler = handler;
		if(loadSdcardAudioConfig) {
			this.config = VoipConfig.loadLoaclJsonConfig(localAudioConfig);
		} else {
			this.config = new VoipConfig(jsonConfig);
		}

		audioManager = (AudioManager) context
				.getApplicationContext()
				.getSystemService(Context.AUDIO_SERVICE);

		//PackageManager packageManager = context.getPackageManager();

		this.audioIO = new VoipAudioIO(audioManager, this.config, context);
		this.audioEffect = new VoipAudioEffect(context);
		this.videoCallback = null;
		this.threadCallback = new ThreadCallback();

		this.orientationListener = new OrientationEventListener(context,
				SensorManager.SENSOR_DELAY_NORMAL) {
			@Override
			public void onOrientationChanged(int orientation) {
				deviceOrientation = orientation;
			}
		};

		if (this.orientationListener.canDetectOrientation() == true) {
			orientationListener.enable();
			VoipLog.d("OrientationEventListener enable");
		} else {
			orientationListener.disable();
			VoipLog.w("OrientationEventListener disable");
		}
	}

	/**
	 * Initialize voip object.
	 *
	 * @param cpu
	 *            cpu level
	 * @param videoCapture
	 *            video capture object used to open/close camera
	 * @param localAspectRatio
	 *            local preview's aspect ratio in integer (e.g. if 4:3, will be 1024*4/3=1365)
	 */
	public void init(int cpu, VoipVideoCapture videoCapture, int localAspectRatio) {
		VoipLog.i("[VoipAndroid][init] Enter");

		if (this.inited) {
			VoipLog.e("[VoipAndroid][init] already inited!");
		} else {
			this.inited = true;
			this.videoCapture = videoCapture;

			String documentPath = Environment.getExternalStorageDirectory()
					.getAbsolutePath();
			contextCallback = createContextCallback();
			context = createContext(contextCallback, documentPath, cpu,
					this.config.audioUseWebRTC, this.config.audioAecOutGain,
					this.config.audioFarendGain,
					this.config.audioAecOutGainGame,
					this.config.audioFarendGainGame,
					this.config.audioAecAggressiveLevel,
					this.config.audioBubbleSpecialMode,
					this.config.colorAlgorithm,
					localAspectRatio,this.config.hardwareSampleRate, this.config.getFrameSize(this.config.hardwareSampleRate), this.config.channelSampleRate);
			audioIO.setClearAudioCacheCallback(this);
			// this.threadCallback.start();
		}

		VoipLog.i("[VoipAndroid][init] Exit");
	}

	/**
	 * Register audio callback.
	 *
	 * @param audioCallback
	 */
	public void registerAudioCallback(AudioCallback audioCallback) {
		this.audioCallback = audioCallback;
		this.audioIO.setAudioCallback(audioCallback);
	}

	/**
	 * Register audio callback.
	 *
	 * @param videoCallback
	 */
	public void registerVideoCallback(VideoCallback videoCallback) {
		this.videoCallback = videoCallback;
	}

	/**
	 * Start voip call.
	 *
	 * @param remote
	 *            remote id string. This is only used in UDP mode
	 * @param mode
	 *            voip mode
	 */
	public void startCall(String remote, int mode) {
		startVoipCall(context, remote, mode);
	}

	/**
	 * Start voip call.
	 *
	 * @param remote
	 *            remote id string. "group" for group call, "live_host" for live show host, "live_guest" for live show guest
	 * @param audioSdp
	 *            voip audio sdp
	 * @param videoSdp
	 *            voip video sdp
	 * @param selfUserIdInGroup
	 *            user ID in group. 1~65535 in group call, 0 for p2p call
	 * @param rtmpServerUrl
	 *            URL address of RTMP server if this is live show host. Otherwise, ""
	 * @param clientPushRtmp
	 *            if TRUE, RTMP stream will be pushed from client instead of UdpServer
	 */
	public void startCall(String remote, String audioSdp, String videoSdp, int selfUserIdInGroup, String rtmpServerUrl, boolean clientPushRtmp) {
		if (! hasSetUserAgent) {
			hasSetUserAgent = true;
			doSetUserAgentProxy(GetUserAgentProxy(3));
		}
		
		String manufacturer = android.os.Build.MANUFACTURER;
		String device = android.os.Build.DEVICE;
		String model = android.os.Build.MODEL;
		VoipLog.i("[VoipAndroid] manufactruer=" + manufacturer + ", device="
				+ device + ", model=" + model);
		startVoipCallWithSdp(context, remote, audioSdp, videoSdp, selfUserIdInGroup, rtmpServerUrl, clientPushRtmp);
	}

	/**
	 * Start voip audio pipeline.
	 *
	 * @param audioParam
	 *            audio parameter in the call
	 */
	public void startAudioPipeline(Activity activity, AudioParameter audioParam, boolean useEarpiece) {
		inAudioCallLock.lock();
		inAudioCall = true;
		audioIO.startPlaying(activity, AUDIO_IO_TYPE.VOIP, null, useEarpiece);
		if (! audioParam.recvOnly) {
			audioIO.startRecording(AUDIO_IO_TYPE.VOIP);
		}
		inAudioCallLock.unlock();

		AudioParameter ap = audioParam;

		if (audioParam == null) {
			ap = new AudioParameter(0, false, false);
		}

		startVoipAudioPipeline(context, ap.bitrate, ap.isGameMode, ap.recvOnly);
		this.doSetSpeakerGain(context, audioIO.getSpeakerGain(ap.isGameMode));
	}

	/**
	 * Start voip video pipeline.
	 *
	 * @param videoParam
	 *            video parameter in the call.
	 * @param liveVideoParam
	 *            video parameter in the live call.
	 */
	public void startVideoPipeline(VideoParameter videoParam, LiveVideoParameter liveVideoParam) {
		VideoParameter vp = videoParam;

		if (videoParam == null) {
			vp = new VideoParameter(0, false);
		}

		if (liveVideoParam == null) {
			startVoipVideoPipeline(context, vp.bitrate, vp.recvOnly);
		} else {
			startVoipVideoPipelineWithLiveParam(context, vp.bitrate, vp.recvOnly,
					liveVideoParam.comp_bottomToTop, liveVideoParam.comp_rightToLeft,
					liveVideoParam.comp_scaleProportion, liveVideoParam.comp_marginVer, liveVideoParam.comp_marginHor,
					liveVideoParam.host_maxBitrate, liveVideoParam.guest_maxBitrate);
		}
	}

	/**
	 * DEPRECATED Stop voip video pipeline.
	 */
	private void stopVideoPipeline() {
		stopVoipVideoPipeline(context);
	}

	/**
	 * DEPRECATED Stop voip audio pipeline.
	 */
	private void stopAudioPipeline() {
		stopVoipAudioPipeline(context);
		this.doSetSpeakerGain(context, audioIO.getSpeakerGain(false));

		inAudioCallLock.lock();
		audioIO.stopRecording();
		audioIO.stopPlaying();
		inAudioCall = false;
		inAudioCallLock.unlock();
	}

	/**
	 * Stop voip call.
	 */
	public void stopCall() {
		stopVoipCall(context);

		inAudioCallLock.lock();
		audioIO.stopRecording();
		audioIO.stopPlaying();
		inAudioCall = false;
		inAudioCallLock.unlock();
	}

	/**
	 * Set audio recv only mode
	 * @param audioRecvOnly recvOnly or not
	 */
	public void setAudioRecvOnly(boolean audioRecvOnly) {
		if (audioRecvOnly) {
			audioIO.stopRecording();
		} else {
			audioIO.startRecording(AUDIO_IO_TYPE.VOIP);
		}
		doSetAudioRecvOnly(context, audioRecvOnly);
	}

	/**
	 * Set video recv only mode
	 * @param videoRecvOnly recvOnly or not
	 */
	public void setVideoRecvOnly(boolean videoRecvOnly) {
		doSetVideoRecvOnly(context, videoRecvOnly);
	}

	/**
	 * Update the count of observer users in group call
	 * @param observerCount observer users count
	 */
	public void updateObserverCountInGroupCall(int observerCount) {
		doUpdateObserverCountInGroupCall(context, observerCount);
	}

	/**
	 * New performer join in the group call
	 * @param userId performer's userId
	 */
	public void performerJoinInGroupCall(int userId) {
		doAddPerformerInGroupCall(context, userId);
	}

	/**
	 * A performer quit the group call
	 * @param userId performer's userId
	 */
	public void performerQuitGroupCall(int userId) {
		doRemovePerformerInGroupCall(context, userId);
	}

	/**
	 * Set local window params
	 * @param sizeLevel window size level
	 * @param aspect window aspect
	 */
	public void setLocalWindow(WindowSizeLevel sizeLevel, int aspect) {
		doSetLocalWindow(context, sizeLevel.index, aspect);
	}

	/**
	 * Set remote window size
	 * @param sizeLevel window size level
	 */
	public void setRemoteWindow(WindowSizeLevel sizeLevel) {
		doSetRemoteWindow(context, sizeLevel.index);
	}

	/**
	 * Uninitialize voip object. This is reverse of init.
	 */
	public void uninit() {
		VoipLog.i("[VoipAndroid][uninit] Enter");

		if (this.inited) {

			destroyContext(context);
			destroyContextCallback(contextCallback);

			context = 0;
			contextCallback = 0;

			this.threadCallback.toStop = true;
			try {
				this.threadCallback.join();
			} catch (InterruptedException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}

			this.inited = false;
		} else {
			VoipLog.e("[VoipAndroid][uninit] not inited!");
		}

		VoipLog.i("[VoipAndroid][uninit] Exit");
	}

	/**
	 * Pause audio devices
	 */
	public void disableAudioIO() {
		audioIO.pause();
	}

	/**
	 * Resume audio devices. Recover audio when the application come back from
	 * background.
	 */
	public void enableAudioIO() {
		inAudioCallLock.lock();
		if (inAudioCall) {
			audioIO.resume();
			clearAudioCache(context);
		}
		inAudioCallLock.unlock();
	}

	/**
	 * Play a mp3 file and mix sound with current audio
	 *
	 * @param uri
	 *            the Content URI of the mp3 data
	 * @param fileName
	 *            mp3 file path
	 * @param cache
	 *            if this file should be cached
	 * @param repeat
	 *            if plays repeatedly
	 * @param crescendo
	 *            crescendo or not
	 */
	public void playMp3File(Activity activity, final Uri uri, String fileName, boolean cache,
							final boolean repeat, boolean crescendo, boolean force, boolean useEarpiece, MP3_TYPE type) {
		inAudioCallLock.lock();
		if (inAudioCall) {
			doPlayMp3File(context, fileName, cache, repeat, crescendo);
		} else {
			if(type == MP3_TYPE.NOTIFICATION) {
				playAudioEffect(fileName,cache,repeat,crescendo,1.0f);

			} else if(type == MP3_TYPE.RINGTONE){

				audioIO.startPlaying(activity, AUDIO_IO_TYPE.RINGTONE, new VoipAudioIO.VoipAudioDataSourceSetter() {
					@Override
					public void doSetDataSource(VoipAudioDevice device) {
						VoipAudioDeviceMediaPlayer mediaPlayer = (VoipAudioDeviceMediaPlayer) device;
						mediaPlayer.setMp3File(uri, repeat);

					}
				}, useEarpiece);
			} else if(type == MP3_TYPE.MUSIC) {
				audioIO.startPlaying(activity, AUDIO_IO_TYPE.PLAYMUSIC, new VoipAudioIO.VoipAudioDataSourceSetter() {
					@Override
					public void doSetDataSource(VoipAudioDevice device) {
						VoipAudioDeviceMediaPlayer mediaPlayer = (VoipAudioDeviceMediaPlayer) device;
						mediaPlayer.setMp3File(uri, repeat);

					}
				}, useEarpiece);
			}
		}
		inAudioCallLock.unlock();
	}

	public void playMp3File(Activity activity, final Uri uri, String fileName, boolean cache,
							final boolean repeat, boolean crescendo, boolean force, boolean useEarpiece) {
		this.playMp3File(activity, uri, fileName, cache, repeat, crescendo, force, useEarpiece, MP3_TYPE.MUSIC);
	}

	public void  playAudioEffect(String path, boolean cache, boolean repeat,
								 boolean crescendo, float volume){
		VoipLog.d("[VoipAndroid][playAudioEffect] begin");
		inAudioCallLock.lock();
		if (inAudioCall) {
			doPlayMp3File(context, path, cache, repeat, crescendo);
		} else {
			audioEffect.StartPlayEffect(path, repeat, volume);
		}
		inAudioCallLock.unlock();
	}

	public void  releaseAudioEffect() {
		VoipLog.d("[VoipAndroid][releaseAudioEffect] begin");
		inAudioCallLock.lock();
		if (inAudioCall) {
			VoipLog.d("[VoipAndroid][releaseAudioEffect] in call no need release soundpool");
		} else {
			audioEffect.end();
		}
		inAudioCallLock.unlock();
	}

	/**
	 * Stop mp3 playing
	 *
	 * @param fileName
	 *            stopped mp3 file path
	 */
	public void stopMp3File(String fileName) {
		doStopMp3File(context, fileName);

		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();
	}

	/**
	 * Stop all mp3 playing
	 */
	public void stopAllMp3Files() {
		doStopAllMp3Files(context);

		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();
	}

	/**
	 * Pause mp3 playing
	 */
	public void pauseMp3() {
		inAudioCallLock.lock();
		if (inAudioCall) {
			doPauseMp3(context);
		}else{
			audioIO.pause();
		}
		inAudioCallLock.unlock();
	}

	/**
	 * Resume mp3 playing
	 */
	public void resumeMp3() {
		inAudioCallLock.lock();
		if (inAudioCall) {
			doResumeMp3(context);
		} else{
			audioIO.resume();
		}
		inAudioCallLock.unlock();
	}

	/**
	 * Start playing stream media.
	 *
	 * @param mediaUrl
	 *            address of media stream
	 * @param cacheDuration
	 *            player cache size in milliseconds
	 * @param callback
	 *            callback to notify code
	 */
	public void startPlayingStreamMedia(Activity activity, String mediaUrl, int cacheDuration,
										StreamMediaPlayerCallback callback, boolean useEarpiece) {
		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.startPlaying(activity, AUDIO_IO_TYPE.VIDEO_MESSAGE, null, useEarpiece);
		}
		inAudioCallLock.unlock();

		streamMediaPlayerCallback = callback;
		doStartPlayingStreamMedia(context, mediaUrl, cacheDuration, 48000); // TODO
	}

	/**
	 * Stop playing stream media.
	 *
	 * @param mediaUrl
	 *            address of media stream
	 */
	public void stopPlayingStreamMedia(String mediaUrl) {
		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();

		doStopPlayingStreamMedia(context, mediaUrl);
		streamMediaPlayerCallback = null;
	}

	/**
	 * Set Max duration of video message
	 *
	 * @param seconds
	 *            max duration in seconds
	 *
	 */
	public void setVideoMessageMaxDuration(int seconds) {
		doSetVideoMessageMaxDuration(context, seconds);
	}

	/**
	 * Prepare recording video message. Must before "startRecordingVideoMessage".
	 *
	 * @param isAvatar
	 *            flag for preparing to record avatar video
	 * @param srcFileDir
	 *            dir path to store src file for AR model, should not null
	 * @param avatarDir
	 *            dir path to store image for avatar, should not null
	 */
	public int prepareRecordingVideoMessage(boolean isAvatar,
											String srcFileDir, String avatarDir) {
		if (isAvatar) {
			VoipLog.e("[prepareRecordingVideoMessage] Avatar video message is not supported in this version!");
			return -1;
		}
		if (isAvatar
				&& (TextUtils.isEmpty(srcFileDir) || TextUtils
				.isEmpty(avatarDir))) {
			return -1;
		}
		doPrepareRecordingVideoMessage(context, isAvatar, srcFileDir, avatarDir);

		return 0;
	}

	/**
	 * Change selected image to avatar image.
	 * @param templateDir
	 * 			  the template path downloaded from server; if null, use the srcImage directly
	 * @param srcImageUrl
	 *			  the file path of the selected image
	 * @param srcFileDir
	 *			  path of avatar model files, should not null
	 * @param avatarDir
	 *			  path of created avatar image, should not null
	 *@param extraData
	 *			  extraData, such as sex, eye position, mouth position, ect
	 */
	public static boolean createAvatarImageWithPicture(String templateDir, String srcImageUrl, String srcFileDir, String avatarDir, String extraData) {
		return doCreateAvatarImageWithPicture(templateDir, srcImageUrl, srcFileDir, avatarDir, extraData);
	}

	/**
	 * Change avatar model befor recording video message.
	 *
	 * @param avatarDir
	 *            dir path to store image for avatar
	 */
	public void changeAvatarTemplate(String avatarDir) {
		doChangeAvatarTemplate(context, avatarDir);
	}

	/**
	 * Start recording video message. Must after "prepareRecordingVideoMessage".
	 *
	 * @param fileName
	 *            file path used to store video message
	 * @param callback
	 *            callback to notify code
	 */
	public void startRecordingVideoMessage(String fileName,
										   VideoMsgRecorderCallback callback) {
		audioIO.startRecording(AUDIO_IO_TYPE.VIDEO_MESSAGE);

		this.videoMsgRecorderCallback = callback;
		doStartRecordingVideoMessage(context, fileName, 16000);
		//doStartRecordingVideoMessage(context, fileName, 44100); // TODO HD_AUDIO
	}

	/**
	 * Stop recording video message. Must before "unprepareRecordingVideoMessage".
	 */
	public void stopRecordingVideoMessage() {
		doStopRecordingVideoMessage(context);
		this.videoMsgRecorderCallback = null;
		audioIO.stopRecording();
	}

	/**
	 * Unprepare recording video message. Must after "stopRecordingVideoMessage".
	 */
	public void unprepareRecordingVideoMessage() {
		doUnprepareRecordingVideoMessage(context);
	}

	/**
	 * resetSpecialAudioDevice
	 *
	 * @param useEarpiece (it can be get from isUseEarpiece() function in VoipAndroidManager.java)
	 *
	 * useEarpiece = true : resetAudioDevice to Earphone
	 *
	 * useEarpiece = false: resetAudioDevice to speaker
	 *
	 */
	public void resetSpecialAudioDevice(boolean useEarpiece){
		if(audioIO != null) {
			VoipLog.d("[VoipAndroid][setPhoneSpeaker] start to set audioIO phone speaker");
			audioIO.resetSpecialAudioDevice(useEarpiece, inAudioCall);
		}
	}

	/**
	 * Start playing video message.
	 *
	 * @param fileName
	 *            video message file path
	 * @param withAudio
	 *            play audio or not
	 * @param callback
	 *            callback to notify code
	 */
	public void startPlayingVideoMessage(Activity activity, String fileName, boolean withAudio,
										 VideoMsgPlayerCallback callback, boolean useEarpiece) {
		if (null == fileName) {
			callback.videoMsgPlayerNotify(fileName, -2);
			return;
		}

		if(withAudio) {
			inAudioCallLock.lock();
			if (!inAudioCall) {
				audioIO.startPlaying(activity, AUDIO_IO_TYPE.VIDEO_MESSAGE, null, useEarpiece);
			}
			inAudioCallLock.unlock();
		}

		videoMsgPlayerCallbacksLock.lock();
		{
			if (! videoMsgPlayerCallbacks.containsKey(fileName)) {
				List<VideoMsgPlayerCallback> callbacks = new ArrayList<VideoMsgPlayerCallback>();
				videoMsgPlayerCallbacks.put(fileName, callbacks);
			}

			List<VideoMsgPlayerCallback> callbacks = videoMsgPlayerCallbacks.get(fileName);
			if (! callbacks.contains(callback)) {
				callbacks.add(callback);
			}

			if ((withAudio || callbacks.size() == 1) && ! videoMsgAllStopped) {
				doStartPlayingVideoMessage(context, fileName, withAudio, 48000); // TODO
			}
		}
		videoMsgPlayerCallbacksLock.unlock();
	}

	/**
	 * Stop playing video message.
	 *
	 * @param fileName
	 *            video message file path
	 * @param remainVideoPreview
	 *            if true, video preview remains playing; if false, stop all
	 * @param stopAudio
	 *            if true, will try to staop audio device
	 * @param callback
	 *            callback to notify code
	 */
	public void stopPlayingVideoMessage(String fileName, boolean remainVideoPreview,
										boolean stopAudio, VideoMsgPlayerCallback callback) {
		inAudioCallLock.lock();
		if (stopAudio && !inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();

		videoMsgPlayerCallbacksLock.lock();
		{
			if (videoMsgPlayerCallbacks.containsKey(fileName)) {
				List<VideoMsgPlayerCallback> callbacks = videoMsgPlayerCallbacks.get(fileName);
				if (callbacks.contains(callback)) {
					boolean lastOne = (1 == callbacks.size());
					if (stopAudio || lastOne) {
						doStopPlayingVideoMessage(context, fileName, remainVideoPreview || (!lastOne));
					}

					if (!remainVideoPreview) {
						callbacks.remove(callback);
						if (callbacks.isEmpty()) {
							videoMsgPlayerCallbacks.remove(fileName);
						}
					}
				}
			}
		}
		videoMsgPlayerCallbacksLock.unlock();
	}

	/**
	 * Get video message duration.
	 *
	 * @param fileName
	 *            video message file path
	 *
	 * @return duration in milliseconds
	 */
	public int getVideoMessageDuration(String fileName) {
		return doGetVideoMessageDuration(context, fileName);
	}

	/**
	 * Stop all video messages
	 *
	 * @param removeCallbacks
	 *            if true, remove all callbacks
	 */
	public void stopAllPlayingVideoMessage(boolean removeCallbacks) {
		VoipLog.d("stopAllPlayingVideoMessage, removeCallbacks=" + removeCallbacks);
		videoMsgAllStopped = true;
		videoMsgPlayerCallbacksLock.lock();
		{
			for(Map.Entry<String,List<VideoMsgPlayerCallback>> entry:videoMsgPlayerCallbacks.entrySet()) {
				doStopPlayingVideoMessage(context, entry.getKey(), false);
			}

			if (removeCallbacks) {
				try {
					videoMsgPlayerCallbacks.clear();
				} catch (Exception e) {
					e.printStackTrace();
				}
			}
		}
		videoMsgPlayerCallbacksLock.unlock();
	}

	/**
	 * Restart all video messages
	 */
	public void restartAllPlayingVideoMessage() {
		VoipLog.d("restartAllPlayingVideoMessage");
		videoMsgAllStopped = false;
		videoMsgPlayerCallbacksLock.lock();
		{
			for(Map.Entry<String,List<VideoMsgPlayerCallback>> entry:videoMsgPlayerCallbacks.entrySet()) {
				doStartPlayingVideoMessage(context, entry.getKey(), false, 16000); // TODO
			}
		}
		videoMsgPlayerCallbacksLock.unlock();
	}

	/**
	 * Set Max duration of audio message
	 *
	 * @param seconds
	 *            max duration in seconds
	 *
	 */
	public void setAudioMessageMaxDuration(int seconds) {
		doSetAudioMessageMaxDuration(context, seconds);
	}

	/**
	 * Start recording audio message.
	 *
	 * @param fileName
	 *            file path used to store audio message
	 * @param format
	 *            audio message format. 0: iSac;  1: Opus;  2: AAC ADTS
	 * @param callback
	 *            callback to notify code
	 */
	public void startRecordingAudioMessage(String fileName, int format,
										   AudioMsgRecorderCallback callback) {
		audioIO.startRecording(AUDIO_IO_TYPE.AUDIO_MESSAGE);

		this.audioMsgRecorderCallback = callback;
		doStartRecordingAudioMessage(context, fileName, format);
	}

	/**
	 * Stop recording audio message.
	 */
	public void stopRecordingAudioMessage() {
		doStopRecordingAudioMessage(context);
		this.audioMsgRecorderCallback = null;
		audioIO.stopRecording();
	}

	/**
	 * Start playing audio message. Only one recorded audio file should be
	 * played once.
	 *
	 * @param fileName
	 *            audio message file path
	 * @param callback
	 *            callback to notify code
	 */
	public void startPlayingAudioMessage(Activity activity, String fileName,
										 AudioMsgPlayerCallback callback, boolean useEarpiece) {
		inAudioCallLock.lock();
		if(!inAudioCall) {
			audioIO.startPlaying(activity, AUDIO_IO_TYPE.AUDIO_MESSAGE, null, useEarpiece);
		}
		inAudioCallLock.unlock();

		this.audioMsgPlayerCallback = callback;
		this.audioMsgFileName = fileName;
		doStartPlayingAudioMessage(context, fileName); // TODO
	}

	/**
	 * Stop playing audio message.
	 */
	public void stopPlayingAudioMessage() {
		inAudioCallLock.lock();
		if(!inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();

		doStopPlayingAudioMessage(context);
		this.audioMsgPlayerCallback = null;
	}

	/**
	 * Get audio message duration.
	 *
	 * @param fileName
	 *            audio message file path
	 *
	 * @return duration in milliseconds
	 */
	public int getAudioMessageDuration(String fileName) {
		return doGetAudioMessageDuration(context, fileName);
	}

	/**
	 * Whether there is wind in audio.
	 *
	 * @return if true, then current user is blow to mic.
	 */
	public boolean audioIsWind() {
		return getAudioIsWind(context);
	}

	/**
	 * Get Audio db.
	 *
	 * @return audio db, range: 0~60
	 */
	public int audioDB() {
		return getAudioDB(context);
	}

	/**
	 * Set game mode.
	 *
	 * @param isGameMode
	 *            game mode or not
	 */
	public void setGameMode(boolean isGameMode) {
		doSetGameMode(context, isGameMode);
		doSetSpeakerGain(context, audioIO.getSpeakerGain(isGameMode));
	}

	/**
	 * Hold video stream temporarily.
	 */
	public void holdVideoPipeline() {
		holdVideo(context);
	}

	/**
	 * Resume video stream.
	 */
	public void resumeVideoPipeline() {
		resumeVideo(context);
	}

	/**
	 * mute mic.
	 */
	public void muteMicrophone() {
		setMuteMicrophone(context, true);
	}

	/**
	 * Unmute mic.
	 */
	public void unmuteMicrophone() {
		setMuteMicrophone(context, false);
	}

	/**
	 * Mute audio output.
	 */
	public void muteSpeaker() {
		setMuteSpeaker(context, true);
	}

	/**
	 * Unmute audio output.
	 */
	public void unmuteSpeaker() {
		setMuteSpeaker(context, false);
	}

	/**
	 * Mute the voice of peer
	 */
	public void muteVoice() {
		setMuteVoice(context, true);
	}

	/**
	 * Resume the voice of peer
	 */
	public void unmuteVoice() {
		setMuteVoice(context, false);
	}

	/**
	 * Set audio output to use speaker or receiver.
	 *
	 * @param use
	 *            use receiver for audio output.
	 */
	public void setUseEarpiece(boolean use) {
		if(audioIO.setUseEarpiece(use) == AUDIO_IO_TYPE.AUDIO_MESSAGE && use) {
			AudioMsgPlayerCallback callback = this.audioMsgPlayerCallback;
			this.audioMsgPlayerCallback = null;
			doStopPlayingAudioMessage(context);
			this.audioMsgPlayerCallback = callback;
			doStartPlayingAudioMessage(context, this.audioMsgFileName); // TODO
		}
		this.doSetSpeakerGain(context, audioIO.getSpeakerGain(null));
		this.setUseWebRTC(!use);
	}

	/**
	 * Set audio output to use bluetooth.
	 *
	 * @param on
	 *            if bluetooth headset is on.
	 */
	public void setBluetooth(boolean on) {
		audioIO.setBluetooth(on);
		this.setUseWebRTC(!on);
	}

	/**
	 * Get voip call statistics.
	 *
	 * @return voip call statistics
	 */
	public VoipStatistics getStatistics() {
		return getStatistics(context);
	}

	/**
	 * Set face beauty mode
	 *
	 * @param mode
	 *            face beauty mode
	 */
	public void setFaceBeautyMode(int mode) {
		setFaceBeautyMode(context, mode);
	}

	/**
	 * Set face beauty light and color.
	 *
	 * @param light
	 *            face beauty light setting
	 * @param color
	 *            face beauty color setting
	 */
	public void setFaceBeautyLightAndColor(int light, int color) {
		setFaceBeautyLightAndColor(context, light, color);
	}

	/**
	 * Set face beauty template strength.
	 *
	 * @param templateStrength
	 *            face beauty template strength
	 */
	public void setFaceBeautyTemplateStrength(int templateStrength) {
		setFaceBeautyTemplateStrength(context, templateStrength);
	}

	/**
	 * Set face beauty strength.
	 *
	 * @param strength
	 *            face beauty strength
	 */
	public void setFaceBeautyStrength(int strength) {
		setFaceBeautyStrength(context, strength);
	}

	/**
	 * set gauss blur enable or disable. If enable, then set the blur radius.
	 *
	 * @param enable
	 *            enable gauss blur or not
	 * @param radius
	 *            blur radius
	 */
	public void setGaussBlur(boolean enable, int radius) {
		setGaussBlur(context, enable, radius);
	}

	/**
	 * Gauss blur an image.
	 *
	 * @param image
	 *            the image object
	 * @param radius
	 *            blur radius
	 */
	public void gaussBlurImage(VoipImage image, int radius) {
		if (image != null) {
			gaussBlurImage(context, image.data, image.width, image.height,
					radius);
		}
	}

	/**
	 * Gauss blur an image in part.
	 *
	 * @param image
	 *            the image object
	 * @param radius
	 *            blur radius
	 * @param left
	 *            coordination left
	 * @param top
	 *            coordination top
	 * @param right
	 *            coordination right
	 * @param bottom
	 *            coordination bottom
	 */
	public void gaussBlurImagePart(VoipImage image, int radius, int left,
								   int top, int right, int bottom) {
		if (image != null) {
			gaussBlurImagePart(context, image.data, image.width, image.height,
					radius, left, top, right, bottom);
		}
	}

	/**
	 * Gauss blur an ARGB image.
	 *
	 * @param data
	 *            ARGB data
	 * @param radius
	 *            blur radius
	 * @param width
	 *            image width
	 * @param height
	 *            image height
	 * @param left
	 *            coordination left
	 * @param top
	 *            coordination top
	 * @param right
	 *            coordination right
	 * @param bottom
	 *            coordination bottom
	 */
	public void gaussBlurARGB(byte[] data, int radius, int width, int height,
							  int left, int top, int right, int bottom) {
		gaussBlurARGB(context, data, width, height, radius, left, top, right,
				bottom);
	}

	/**
	 * Set QoS parameters
	 *
	 * @param qosJson
	 */
	public void setQosParam(String qosJson) {
		setQosParam(context, qosJson);
	}

	/*
	 * set VOIP log level
	 *
	 * @param log level -1: DISABLE; 0:ERROR; 1:WARNING; 2:INFO; 3:DEBUG;
	 */
	public static void setLogLevel(int logLevel) {
		VoipLog.logLevel = logLevel;
		doSetLogLevel(logLevel);
	}

	/*
	 * enable VOIP processor
	 *
	 * @param processorID
	 *      2:   audio postprocess
	 *      101: video prescale
	 *      102: video postscale
	 *      103: face beauty
	 */
	public void enableProcessor(int processorID, boolean enable) {
		doEnableProcessor(context, processorID, enable);
	}

	/*
	 * get VOIP version
	 *
	 * @return version string
	 */
	public static native String version();

	/**
	 * Do bencnmark to calculate the CPU level. This method will take a long
	 * time to return and should be called in a separate thread.
	 *
	 * @return cpu level
	 */
	public static native int benchmark();

	/**
	 * Get current decoded or preview image.
	 *
	 * @param sourceID
	 *            the source id to presend which stream the image will be got
	 * @return the image object
	 */
	public static native VoipImage getImage(String sourceID);

	public static native void yuv_to_rgb565(byte[] rgb565, byte[] yuv,
											int width, int height);

	public static native void stretch_yv12(byte[] dst, byte[] src, int width,
										   int height, float fitRate);

	public static native void yuv_rotate90(byte[] dst, byte[] src, int width,
										   int height, boolean mirror);

	public static native void yuv_rotate270(byte[] dst, byte[] src, int width,
											int height, boolean mirror);

	public static native void yuv_rotate180(byte[] dst, byte[] src, int width,
											int height, boolean mirror);

	/**
	 * Generate an UIImage with the given VoipImage object.
	 *
	 * @param voipImage
	 *            the source id to presend which stream the image will be got
	 * @return the UIImage object
	 */
	public Bitmap createBitmapWithVoipImage(VoipImage voipImage) {
		byte[] yuv = new byte[voipImage.width * voipImage.height * 3 / 2];
		float tempLift = 1.0f + voipImage.facelift;
		stretch_yv12(yuv, voipImage.data, voipImage.width, voipImage.height,
				tempLift);
		byte[] rgb565 = new byte[voipImage.width * voipImage.height * 2];
//		if (0 == voipImage.rotation) {
//			byte[] yuv420 = new byte[voipImage.width * voipImage.height * 3 / 2];
//			yuv_rotateB90(yuv420, yuv, voipImage.width, voipImage.height);
//			int tempNum = voipImage.width;
//	        voipImage.width = voipImage.height;
//	        voipImage.height = tempNum;
//	        yuv_to_rgb565(rgb565, yuv420, voipImage.width, voipImage.height);
//		} else if (2 == voipImage.rotation) {
//			byte[] yuv420 = new byte[voipImage.width * voipImage.height * 3 / 2];
//			yuv_rotate90(yuv420, yuv, voipImage.width, voipImage.height);
//			int tempNum = voipImage.width;
//	        voipImage.width = voipImage.height;
//	        voipImage.height = tempNum;
//	        yuv_to_rgb565(rgb565, yuv420, voipImage.width, voipImage.height);
//		} else if (1 == voipImage.rotation) {
//			byte[] yuv420 = new byte[voipImage.width * voipImage.height * 3 / 2];
//			yuv_rotate180(yuv420, yuv, voipImage.width, voipImage.height);
//			yuv_to_rgb565(rgb565, yuv420, voipImage.width, voipImage.height);
//		} else {
		yuv_to_rgb565(rgb565, yuv, voipImage.width, voipImage.height);
//		}
		Bitmap bmp = Bitmap.createBitmap(voipImage.width, voipImage.height,
				Bitmap.Config.RGB_565);
		ByteBuffer buf = ByteBuffer.wrap(rgb565); // data is my array
		if (buf == null || bmp == null) {
			return null;
		}
		bmp.copyPixelsFromBuffer(buf);
		return bmp;
	}

	/*
	 * 减少GC，更新上面一个方法
	 * */
	public Bitmap createBitmapWithVoipImage(VoipImage voipImage, byte[] yuv,
											byte[] rgb565, Bitmap bmp) {
		if (yuv == null
				|| yuv.length != voipImage.width * voipImage.height * 3 / 2) {
			yuv = new byte[voipImage.width * voipImage.height * 3 / 2];
		}
		float tempLift = (0 == voipImage.rotation || 2 == voipImage.rotation) ? 1.0f - voipImage.facelift : 1.0f + voipImage.facelift;
		stretch_yv12(yuv, voipImage.data, voipImage.width, voipImage.height,
				tempLift);
		if (rgb565 == null
				|| rgb565.length != voipImage.width * voipImage.height * 2) {
			rgb565 = new byte[voipImage.width * voipImage.height * 2];
		}
		yuv_to_rgb565(rgb565, yuv, voipImage.width, voipImage.height);
		if (bmp == null || bmp.getWidth() != voipImage.width
				|| bmp.getHeight() != voipImage.height
				|| !bmp.getConfig().equals(Bitmap.Config.RGB_565)) {
			bmp = Bitmap.createBitmap(voipImage.width, voipImage.height,
					Bitmap.Config.RGB_565);
		}
		ByteBuffer buf = ByteBuffer.wrap(rgb565); // data is my array
		if (buf == null || bmp == null) {
			return null;
		}
		bmp.copyPixelsFromBuffer(buf);
		return bmp;
	}

	private void nativeCallbackRemoteAudioChanged(boolean enable) {
		VoipLog.i("RemoteAudioChanged enable=" + enable);
		if (audioCallback != null) {
			audioCallback.remoteAudioChanged(enable);
		}
	}

	private void nativeCallbackRemoteVideoChanged(boolean enable) {
		VoipLog.i("RemoteVideoChanged enable:" + enable);
		if (videoCallback != null) {
			videoCallback.remoteVideoChanged(enable);
		}
	}

	private void nativeCallbackLocalVideoChanged(boolean enable) {
		VoipLog.i("LocalVideoChanged enable:" + enable);
		if (videoCallback != null) {
			videoCallback.localVideoChanged(enable);
		}
	}

	private void nativeCallbackStreamMediaPlayerNotify(String streamUrl, int code) {
		VoipLog.w("StreamMediaPlayerNotify: streamUrl=" + streamUrl + ", code=" + code);
		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();

		streamMediaPlayerCallback.streamMediaPlayerNotify(streamUrl, code);
	}

	private void nativeCallbackStreamMediaDecodedImage(String streamUrl, byte[] data,
													   int width, int height, boolean isDarkness, boolean isFront,
													   int rotation, int faceLift, int userId) {
		VoipImage image = new VoipImage();
		image.data = data;
		image.width = width;
		image.height = height;
		image.isDarkness = isDarkness;
		image.isFrontCamera = isFront;
		image.rotation = rotation;
		// 10~30 -> -0.05~0.15
		image.facelift = (1.0f * faceLift - 15.0f) / 100.0f;
		image.userId = userId;
		streamMediaPlayerCallback.streamMediaDecodedImage(image);
	}

	private void nativeCallbackVideoMsgRecorderNotify(int code) {
		VoipLog.w("VideoMsgRecorderNotify: " + code);
		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.stopRecording();
		}
		inAudioCallLock.unlock();
		if (videoMsgRecorderCallback != null) {
			videoMsgRecorderCallback.videoMsgRecorderNotify(code);
		}
	}

	private void nativeCallbackVideoMsgData(byte[] data, int len,
											boolean isFileTailer) {
		if (videoMsgRecorderCallback != null) {
			videoMsgRecorderCallback.videoMsgData(data, len, isFileTailer);
		}
	}

	private void nativeCallbackVideoMsgFirstFrame(byte[] data,
												  int width, int height, boolean isDarkness, boolean isFront,
												  int rotation, int faceLift, int userId) {
		VoipImage image = new VoipImage();
		image.data = data;
		image.width = width;
		image.height = height;
		image.isDarkness = isDarkness;
		image.isFrontCamera = isFront;
		image.rotation = rotation;
		// 10~30 -> -0.05~0.15
		image.facelift = (1.0f * faceLift - 15.0f) / 100.0f;
		image.userId = userId;

		if (videoMsgRecorderCallback != null) {
			videoMsgRecorderCallback.videoMsgFirstFrame(image);
		}
	}

	private void nativeCallbackVideoMsgPlayerNotify(String fileName, int code) {
		VoipLog.w("VideoMsgPlayerNotify: fileName=" + fileName + ", code=" + code);
		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();

		videoMsgPlayerCallbacksLock.lock();
		if (videoMsgPlayerCallbacks.containsKey(fileName)) {
			List<VideoMsgPlayerCallback> callbacks = videoMsgPlayerCallbacks.get(fileName);
			for (int i = 0; i < callbacks.size(); i++) {
				VideoMsgPlayerCallback cb = callbacks.get(i);
				cb.videoMsgPlayerNotify(fileName, code);
			}
		}
		videoMsgPlayerCallbacksLock.unlock();
	}

	private void nativeCallbackVideoMsgDecodedImage(String fileName, byte[] data,
													int width, int height, boolean isDarkness, boolean isFront,
													int rotation, int faceLift, int userId) {
		videoMsgPlayerCallbacksLock.lock();
		if (videoMsgPlayerCallbacks.containsKey(fileName)) {
			VoipImage image = new VoipImage();
			image.data = data;
			image.width = width;
			image.height = height;
			image.isDarkness = isDarkness;
			image.isFrontCamera = isFront;
			image.rotation = rotation;
			// 10~30 -> -0.05~0.15
			image.facelift = (1.0f * faceLift - 15.0f) / 100.0f;
			image.userId = userId;

			List<VideoMsgPlayerCallback> callbacks = videoMsgPlayerCallbacks.get(fileName);
			for (int i = 0; i < callbacks.size(); i++) {
				VideoMsgPlayerCallback cb = callbacks.get(i);
				cb.videoMsgDecodedImage(image);
			}
		}
		videoMsgPlayerCallbacksLock.unlock();
	}

	private void nativeCallbackAudioMsgRecorderNotify(int code) {
		VoipLog.w("AudioMsgRecorderNotify: " + code);
		inAudioCallLock.lock();
		if (!inAudioCall) {
			audioIO.stopRecording();
		}
		inAudioCallLock.unlock();

		if (audioMsgRecorderCallback != null) {
			audioMsgRecorderCallback.audioMsgRecorderNotify(code);
		}
	}

	private void nativeCallbackAudioMsgData(byte[] data, int len,
											boolean isFileHeader) {
		if (audioMsgRecorderCallback != null) {
			audioMsgRecorderCallback.audioMsgData(data, len, isFileHeader);
		}
	}

	private void nativeCallbackAudioMsgPlayerNotify(int code) {
		VoipLog.w("AudioMsgPlayerNotify: " + code);
		inAudioCallLock.lock();
		if(!inAudioCall) {
			audioIO.stopPlaying();
		}
		inAudioCallLock.unlock();

		if (audioMsgPlayerCallback != null) {
			audioMsgPlayerCallback.audioMsgPlayerNotify(code);
		}
	}

	private void nativeCallbackVideoFrame(boolean isLocal, byte[] data,
										  int width, int height, boolean isDarkness, boolean isFront,
										  int rotation, int faceLift, int userId) {
		VoipImage image = new VoipImage();
		image.data = data;
		image.width = width;
		image.height = height;
		image.isDarkness = isDarkness;
		image.isFrontCamera = isFront;
		image.rotation = rotation;
		// 10~30 -> -0.05~0.15
		image.facelift = (1.0f * faceLift - 15.0f) / 100.0f;
		image.userId = userId;

		if (videoCallback != null) {
			if (isLocal) {
				videoCallback.previewImage(image);
			} else {
				videoCallback.decodedImage(image);
			}
		}
	}

	private void nativeCallbackAudioAnalysis(boolean isWind, int dB) {
		if (audioCallback != null) {
			audioCallback.audioIsWind(isWind);
			audioCallback.audioDB(dB);
		}
	}

	private native long createContextCallback();

	private native void destroyContextCallback(long callback);

	private native long createContext(long callback, String documentPath,
									  int cpu, boolean enableAec, float aecOutGain, float decOutGain,
									  float aecOutGainGameMode, float decOutGainGameMode,
									  int aecAggressiveLevel, int bubbleSpecialModel,
									  int faceBeautyDeviceID, int localAspectRatio,int recordSampleRate, int recordInputSize, int channelSampleRate);

	private native void doChangeCallAvatar(long context, boolean useAvatar, String srcFileDir, String avatarImagePath);

	private native void destroyContext(long context);

	public static native String[] getVoipSdp();

	private native void startVoipCall(long context, String remote, int mode);

	private native void startVoipCallWithSdp(long context, String remote,
											 String audioSdp, String videoSdp, int selfUserId, String rtmpServerUrl, boolean clientPushRtmp);

	private native void startVoipAudioPipeline(long context, int audioBitrate,
											   boolean isGameMode, boolean audioRecvOnly);

	private native void startVoipVideoPipeline(long context, int videoBitrate,
											   boolean videoRecvOnly);

	private native void startVoipVideoPipelineWithLiveParam(long context, int videoBitrate, boolean videoRecvOnly,
															boolean comp_bToT, boolean comp_rToL, float comp_scale, float comp_mgVer, float comp_mgHor,
															int host_maxBr, int guest_maxBr);

	private native void stopVoipVideoPipeline(long context);

	private native void stopVoipAudioPipeline(long context);

	private native void stopVoipCall(long context);

	private native void doSetAudioRecvOnly(long context, boolean audioRecvOnly);

	private native void doSetVideoRecvOnly(long context, boolean videoRecvOnly);

	private native void doUpdateObserverCountInGroupCall(long context, int observerCount);

	private native void doAddPerformerInGroupCall(long context, int userId);

	private native void doRemovePerformerInGroupCall(long context, int userId);

	private native void doSetLocalWindow(long context, int sizeLevel, int aspect);

	private native void doSetRemoteWindow(long context, int sizeLevel);

	private native void putVideoData(byte[] data, int width, int height,
									 boolean front, int angle, int faceLift);

	private native VoipStatistics getStatistics(long context);

	private native void doPlayMp3File(long context, String fileName,
									  boolean cache, boolean repeat, boolean crescendo);

	private native void doStopMp3File(long context, String fileName);

	private native void doStopAllMp3Files(long context);

	private native void doPauseMp3(long context);

	private native void doResumeMp3(long context);

	private native void doStartPlayingStreamMedia(long context, String mediaUrl, int cacheDuration, int audioSampleRate);

	private native void doStopPlayingStreamMedia(long context, String mediaUrl);

	private native void doSetVideoMessageMaxDuration(long context, int seconds);

	private native void doPrepareRecordingVideoMessage(long context, boolean isAvatar, String avatarModelPath, String avatarImagePath);

	private static native boolean doCreateAvatarImageWithPicture(String templateDir, String srcImageUrl, String srcFileDir, String avatarDir, String extraData);

	private native void doChangeAvatarTemplate(long context, String avatarImagePath);

	private native void doStartRecordingVideoMessage(long context, String fileName, int audioSampleRate);

	private native void doStopRecordingVideoMessage(long context);

	private native void doUnprepareRecordingVideoMessage(long context);

	private native void doStartPlayingVideoMessage(long context, String fileName, boolean withAudio, int audioSampleRate);

	private native void doStopPlayingVideoMessage(long context, String fileName, boolean remainVideoPreview);

	private native int doGetVideoMessageDuration(long context, String fileName);

	private native void doSetAudioMessageMaxDuration(long context, int seconds);

	private native void doStartRecordingAudioMessage(long context,
													 String fileName, int format);

	private native void doStopRecordingAudioMessage(long context);

	private native void doStartPlayingAudioMessage(long context, String fileName);

	private native void doStopPlayingAudioMessage(long context);

	private native int doGetAudioMessageDuration(long context, String fileName);

	private native void holdVideo(long context);

	private native void resumeVideo(long context);

	private native void clearAudioCache(long context);

	private native boolean getAudioIsWind(long context);

	private native int getAudioDB(long context);

	private native void doSetGameMode(long context, boolean isGameMode);

	private native void setMuteMicrophone(long context, boolean enable);

	private native void setMuteSpeaker(long context, boolean enable);

	private native void setMuteVoice(long context, boolean enable);

	private native void setFaceBeautyMode(long context, int mode);

	private native void setFaceBeautyLightAndColor(long context, int light,
												   int color);

	private native void setFaceBeautyTemplateStrength(long context,
													  int templateStrength);

	private native void setFaceBeautyStrength(long context, int strength);

	private native void setGaussBlur(long context, boolean enable, int radius);

	private native void gaussBlurImage(long context, byte[] data, int width,
									   int height, int radius);

	private native void gaussBlurImagePart(long context, byte[] data,
										   int width, int height, int radius, int left, int top, int right,
										   int bottom);

	private native void gaussBlurARGB(long context, byte[] data, int width,
									  int height, int radius, int left, int top, int right, int bottom);

	private native void setQosParam(long context, String qosJson);

	private native static void doSetLogLevel(int logLevel);

	private native void doSetSpeakerGain(long context, float gain);

	private native void doSetUseWebRTC(long context, boolean useWebRTC);
	private native void doSetRecordSampleRate(long context, int recordSampleRate, int recordInputsize, int channelSampleRate);

	private native void doEnableProcessor(long context, int processorID, boolean enable);

	private native void doSetAecOutGain(long context, float gain);

	public native long GetUserAgentProxy(int appID);
	public native static void doSetUserAgentProxy(long userAgentProxy);

	@Override
	public void onClearAudioCache() {
		if (context != 0) {
			clearAudioCache(context);
		}
	}

	@Override
	public void onVideoCaptureData(byte[] data, int width, int height,
								   boolean front, int angle, float faceLift) {
		// 0/180 -> 1, 90/270 -> -1
		double rad = Math.cos(deviceOrientation * 4 * 3.14159f / 360);
		int faceliftInt = (int) (15 + faceLift * 100 * rad + 0.5); // round
		putVideoData(data, width, height, front, angle, faceliftInt);
	}


	public void updateConfig(JSONObject json) {
		if(loadSdcardAudioConfig) {
			this.config = VoipConfig.loadLoaclJsonConfig(localAudioConfig);
		} else {
			this.config = new VoipConfig(json);
		}



		if(this.audioIO != null) {
			this.audioIO.setConfig(this.config);
		}

		this.setUseWebRTC(this.config.audioUseWebRTC);
		this.doSetAecOutGain(context, this.config.audioAecOutGain);
		this.setRecordSampleRate();
	}

	private void setUseWebRTC(boolean use) {
		if(this.config.audioUseWebRTC) {
			VoipLog.d("[VoipAndroid] [setUseWebRTC] useRTC = " + use);
			this.doSetUseWebRTC(context,this.config.audioUseWebRTC);
		}
	}


	private void setRecordSampleRate() {

			VoipLog.d("[VoipAndroid] [setRecordSampleRate] audioDevice = " + this.config.audioDevice+",samplerate= "+ this.config.hardwareSampleRate);

			this.doSetRecordSampleRate(context, this.config.hardwareSampleRate, this.config.getFrameSize(this.config.hardwareSampleRate), this.config.channelSampleRate);
	}

	private static final String sdcardSoPath = "/sdcard/libvoip_android.so";
	public static void loadVoipLibrary() {

		File soFile = new File(sdcardSoPath);
		if(soFile.exists()) {
			loadSdcardLibrary = true;

		} else {
			System.loadLibrary("voip_android");
		}
	}

	private static JSONObject localAudioConfig = null;

	private static final String audioConfigPath = "/sdcard/audio_config.json";
	private static JSONObject loadAudioConfig() {

		StringBuilder sb = new StringBuilder();
		File configFile = new File(audioConfigPath);
		if(configFile.exists()) {

			VoipLog.w("===============  Warnning !!!  ==============");
			VoipLog.w("==  Load Audio config from sdcard for TEST ==");
			VoipLog.w("=============================================");

			try {
				BufferedReader br = new BufferedReader(new FileReader(configFile));
				String line;
				while ((line = br.readLine()) != null) {
					sb.append(line);
				}

				JSONObject obj = JSON.parseObject(sb.toString());
				return obj;
			} catch (Exception e) {
				e.printStackTrace();
				return null;
			}
		} else {
			return null;
		}
	}

	private void loadVoipLibraryfromSDCard(Context context) {

		VoipLog.w("===============  Warnning !!!  ==============");
		VoipLog.w("==  Load VOIP library from sdcard for TEST ==");
		VoipLog.w("=============================================");

		String dataFilePath = context.getFilesDir().getAbsolutePath();
		String copySoPath = dataFilePath + File.separator + ".so";
		File mkSoPath = new File(copySoPath);
		mkSoPath.mkdirs();
		String copySoFile = copySoPath + File.separator + "libvoip_android.so";

		VoipLog.d("Copy " + sdcardSoPath + " to " + copySoFile);
		File toFile = new File(copySoFile);
		if(toFile.exists()) {
			toFile.delete();
		}

		FileInputStream fi = null;
		FileOutputStream fo = null;
		FileChannel in = null;
		FileChannel out = null;

		try {
			fi = new FileInputStream(sdcardSoPath);
			fo = new FileOutputStream(copySoFile);
			in = fi.getChannel();
			out = fo.getChannel();
			in.transferTo(0, in.size(), out);

		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			try {
				fi.close();
				in.close();
				fo.close();
				out.close();
			} catch (IOException e) {
				e.printStackTrace();
			}
		}

		System.load(copySoFile);

		VoipLog.d(" VOIP library load from sdcard finished ");

	}

	static {
		System.loadLibrary("hwnetwork");
		//System.loadLibrary("opencv_java");
		//System.loadLibrary("avatar");
		VoipAndroid.loadVoipLibrary();

		//load local audio config for test
		localAudioConfig = loadAudioConfig();
		if(localAudioConfig != null) {
			loadSdcardAudioConfig = true;
		}

	}

}
