package me.chatgame.voip;

/**
 * Created by wangkui on 16/2/17.
 */
import android.content.Context;
import android.media.AudioAttributes;
import android.media.AudioManager;
import android.media.SoundPool;
import android.net.Uri;
import android.os.Build;
import android.util.Log;
import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;

public class VoipAudioEffect {
    // ===========================================================
    // Constants
    // ===========================================================

    private static final String TAG = "VoipAudioEffect";

    // ===========================================================
    // Fields
    // ===========================================================

    private final Context mContext;
    private SoundPool mSoundPool;
    private float mLeftVolume;
    private float mRightVolume;

    // sound path and stream ids map
    // a file may be played many times at the same time
    // so there is an array map to a file path
    private final HashMap<String, ArrayList<Integer>> mPathStreamIDsMap = new HashMap<String, ArrayList<Integer>>();

    private final HashMap<String, Integer> mPathSoundIDMap = new HashMap<String, Integer>();

    private ConcurrentHashMap<Integer, SoundInfoForLoadedCompleted>  mPlayWhenLoadedEffects =
            new ConcurrentHashMap<Integer, SoundInfoForLoadedCompleted>();

    private static final int MAX_SIMULTANEOUS_STREAMS_DEFAULT = 5;
    private static final int MAX_SIMULTANEOUS_STREAMS_I9100 = 3;
    private static final float SOUND_RATE = 1.0f;
    private static final int SOUND_PRIORITY = 1;
    private static final int SOUND_QUALITY = 5;

    private final static int INVALID_SOUND_ID = -1;
    private final static int INVALID_STREAM_ID = -1;
    // ===========================================================
    // Constructors
    // ===========================================================


    public VoipAudioEffect(final Context context) {
        this.mContext = context;
        this.initData();
    }

    private void initData() {

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP) {
            this.mSoundPool = new SoundPool.Builder()
                    .setMaxStreams(MAX_SIMULTANEOUS_STREAMS_DEFAULT)
                    .setAudioAttributes(new AudioAttributes.Builder()
                            .setUsage(AudioAttributes.USAGE_NOTIFICATION)
                            .setContentType(AudioAttributes.CONTENT_TYPE_SONIFICATION)
                            .build())
                    .build();
        } else {
            this.mSoundPool = new SoundPool(VoipAudioEffect.MAX_SIMULTANEOUS_STREAMS_DEFAULT, AudioManager.STREAM_NOTIFICATION, VoipAudioEffect.SOUND_QUALITY);
        }
        this.mSoundPool.setOnLoadCompleteListener(new OnLoadCompletedListener());
        this.mLeftVolume = 0.5f;
        this.mRightVolume = 0.5f;
    }

    public int preloadEffect(final String path) {
        Integer soundID = this.mPathSoundIDMap.get(path);

        if (soundID == null) {
            soundID = this.createSoundIDFromAsset(path);
            // save value just in case if file is really loaded
            if (soundID != VoipAudioEffect.INVALID_SOUND_ID) {
                VoipLog.d("[VoidAudioEffect][preloadEffect]soundID add : " + soundID);
                this.mPathSoundIDMap.put(path, soundID);
            }
        }

        return soundID;
    }

    public void unloadEffect(final String path) {
        // stop effects
        final ArrayList<Integer> streamIDs = this.mPathStreamIDsMap.get(path);
        if (streamIDs != null) {
            for (final Integer steamID : streamIDs) {
                this.mSoundPool.stop(steamID);
            }
        }
        this.mPathStreamIDsMap.remove(path);
        // unload effect
        final Integer soundID = this.mPathSoundIDMap.get(path);
        if(soundID != null) {
            this.mSoundPool.unload(soundID);
            this.mPathSoundIDMap.remove(path);
        }
    }

    public void StartPlayEffect(String path, boolean repeat, float volume){
        playEffect(path, repeat, 1, 0, volume);
    }

    private static int LOAD_TIME_OUT = 500;

    public int playEffect(final String path, final boolean loop, float pitch, float pan, float gain){
        Integer soundID = this.mPathSoundIDMap.get(path);
        int streamID;
        VoipLog.d("[VoipAudioEffect][playEffect] START");
        if (soundID != null) {
            // parameters; pan = -1 for left channel, 1 for right channel, 0 for both channels

            // play sound
            streamID = this.doPlayEffect(path, soundID, loop, pitch, pan, gain);
        } else {
            // the effect is not prepared
            soundID = this.preloadEffect(path);
            if (soundID == VoipAudioEffect.INVALID_SOUND_ID) {
                // can not preload effect
                return VoipAudioEffect.INVALID_SOUND_ID;
            }

            SoundInfoForLoadedCompleted info = new SoundInfoForLoadedCompleted(path, loop, pitch, pan, gain);
            mPlayWhenLoadedEffects.putIfAbsent(soundID, info);

            synchronized(info) {
                try {
                    info.wait(LOAD_TIME_OUT);
                }
                catch (Exception e) {
                    e.printStackTrace();
                }
            }
            streamID = info.effectID;
            mPlayWhenLoadedEffects.remove(soundID);
        }

        return streamID;
    }

    public void stopEffect(final int steamID) {
        this.mSoundPool.stop(steamID);

        // remove record
        for (final String pPath : this.mPathStreamIDsMap.keySet()) {
            if (this.mPathStreamIDsMap.get(pPath).contains(steamID)) {
                this.mPathStreamIDsMap.get(pPath).remove(this.mPathStreamIDsMap.get(pPath).indexOf(steamID));
                break;
            }
        }
    }

    public void pauseEffect(final int steamID) {
        this.mSoundPool.pause(steamID);
    }

    public void resumeEffect(final int steamID) {
        this.mSoundPool.resume(steamID);
    }

    public void pauseAllEffects() {
        if (!this.mPathStreamIDsMap.isEmpty()) {
            final Iterator<Entry<String, ArrayList<Integer>>> iter = this.mPathStreamIDsMap.entrySet().iterator();
            while (iter.hasNext()) {
                final Entry<String, ArrayList<Integer>> entry = iter.next();
                for (final int steamID : entry.getValue()) {
                    this.mSoundPool.pause(steamID);
                }
            }
        }
    }

    public void resumeAllEffects() {
        // can not only invoke SoundPool.autoResume() here, because
        // it only resumes all effects paused by pauseAllEffects()
        if (!this.mPathStreamIDsMap.isEmpty()) {
            final Iterator<Entry<String, ArrayList<Integer>>> iter = this.mPathStreamIDsMap.entrySet().iterator();
            while (iter.hasNext()) {
                final Entry<String, ArrayList<Integer>> entry = iter.next();
                for (final int steamID : entry.getValue()) {
                    this.mSoundPool.resume(steamID);
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    public void stopAllEffects() {
        // stop effects
        if (!this.mPathStreamIDsMap.isEmpty()) {
            final Iterator<?> iter = this.mPathStreamIDsMap.entrySet().iterator();
            while (iter.hasNext()) {
                final Entry<String, ArrayList<Integer>> entry = (Entry<String, ArrayList<Integer>>) iter.next();
                for (final int steamID : entry.getValue()) {
                    this.mSoundPool.stop(steamID);
                }
            }
        }

        // remove records
        this.mPathStreamIDsMap.clear();
    }

    public float getEffectsVolume() {
        return (this.mLeftVolume + this.mRightVolume) / 2;
    }

    public void setEffectsVolume(float volume) {
        // volume should be in [0, 1.0]
        if (volume < 0) {
            volume = 0;
        }
        if (volume > 1) {
            volume = 1;
        }

        this.mLeftVolume = this.mRightVolume = volume;

        // change the volume of playing sounds
        if (!this.mPathStreamIDsMap.isEmpty()) {
            final Iterator<Entry<String, ArrayList<Integer>>> iter = this.mPathStreamIDsMap.entrySet().iterator();
            while (iter.hasNext()) {
                final Entry<String, ArrayList<Integer>> entry = iter.next();
                for (final int steamID : entry.getValue()) {
                    this.mSoundPool.setVolume(steamID, this.mLeftVolume, this.mRightVolume);
                }
            }
        }
    }

    private void unloadAllSoundID(){
        if(!this.mPathSoundIDMap.isEmpty()){
            final Iterator<Entry<String,Integer>> iter = this.mPathSoundIDMap.entrySet().iterator();
            while(iter.hasNext()){
                final Entry<String ,Integer> entry = iter.next();
                VoipLog.d("[VoipAudioEffect][unloadAllSoundID]unload SoundID = " + entry.getValue());
                this.mSoundPool.unload(entry.getValue());
            }
        }
    }

    public void end() {
        VoipLog.d("[VoipAudioEffect][end] begin !");
        unloadAllSoundID();
        this.mSoundPool.release();
        this.mPathStreamIDsMap.clear();
        this.mPathSoundIDMap.clear();
        this.mPlayWhenLoadedEffects.clear();

        this.mLeftVolume = 0.5f;
        this.mRightVolume = 0.5f;

        this.initData();
    }

    public int createSoundIDFromAsset(final String path) {
        int soundID;

        try {
            if (path.startsWith("/")) {
                soundID = this.mSoundPool.load(path, 1);
            } else {
               soundID = this.mSoundPool.load(this.mContext.getAssets().openFd(path), 0);
            }
        } catch (final Exception e) {
            soundID = VoipAudioEffect.INVALID_SOUND_ID;
            Log.e(VoipAudioEffect.TAG, "error: " + e.getMessage(), e);
        }

        // mSoundPool.load returns 0 if something goes wrong, for example a file does not exist
        if (soundID == 0) {
            soundID = VoipAudioEffect.INVALID_SOUND_ID;
        }

        return soundID;
    }

    private float clamp(float value, float min, float max) {
        return Math.max(min, (Math.min(value, max)));
    }

    private int doPlayEffect(final String path, final int soundId, final boolean loop, float pitch, float pan, float gain) {
        float leftVolume = this.mLeftVolume * gain * (1.0f - this.clamp(pan, 0.0f, 1.0f));
        float rightVolume = this.mRightVolume * gain * (1.0f - this.clamp(-pan, 0.0f, 1.0f));
        float soundRate = this.clamp(SOUND_RATE * pitch, 0.5f, 2.0f);
        VoipLog.d("[VoipAudioEffect][doPlayEffect] START");
        // play sound
        int streamID = this.mSoundPool.play(soundId, this.clamp(leftVolume, 0.0f, 1.0f), this.clamp(rightVolume, 0.0f, 1.0f), VoipAudioEffect.SOUND_PRIORITY, loop ? -1 : 0, soundRate);
        // record stream id
        ArrayList<Integer> streamIDs = this.mPathStreamIDsMap.get(path);
        if (streamIDs == null) {
            streamIDs = new ArrayList<>();
            this.mPathStreamIDsMap.put(path, streamIDs);
        }
        streamIDs.add(streamID);

        return streamID;
    }

    public void onEnterBackground(){
        this.mSoundPool.autoPause();
    }

    public void onEnterForeground(){
        this.mSoundPool.autoResume();
    }

    // ===========================================================
    // Inner and Anonymous Classes
    // ===========================================================

    public class SoundInfoForLoadedCompleted {
        public boolean isLoop;
        public float pitch;
        public float pan;
        public float gain;
        public String path;
        public int effectID;

        public SoundInfoForLoadedCompleted(String path, boolean isLoop, float pitch, float pan, float gain) {
            this.path = path;
            this.isLoop = isLoop;
            this.pitch = pitch;
            this.pan = pan;
            this.gain = gain;
            effectID = VoipAudioEffect.INVALID_SOUND_ID;
        }
    }

    public class OnLoadCompletedListener implements SoundPool.OnLoadCompleteListener {

        @Override
        public void onLoadComplete(SoundPool soundPool, int sampleId, int status) {
            if (status == 0)
            {
                SoundInfoForLoadedCompleted info =  mPlayWhenLoadedEffects.get(sampleId);
                if (info != null) {
                    info.effectID = doPlayEffect(info.path, sampleId, info.isLoop, info.pitch, info.pan, info.gain);
                    synchronized (info) {
                        info.notifyAll();
                    }
                }
            }
        }
    }


}

