package me.chatgame.voip;

import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

import android.R.integer;
import android.app.admin.DevicePolicyManager;
import android.content.Context;
import android.hardware.Camera;
import android.hardware.Camera.CameraInfo;
import android.hardware.Camera.Parameters;
import android.hardware.Camera.Size;
import android.view.SurfaceHolder;
import android.view.SurfaceView;
import android.view.ViewGroup;
import android.widget.FrameLayout;

/**
 * Video capturer class controls the video camera start and stop.
 */
public class VoipVideoCapture implements Camera.PreviewCallback,
		SurfaceHolder.Callback {

	public static class CaptureParam {
		public int width = MAX_SUPPORTED_WIDTH;
		public int height = MAX_SUPPORTED_HEIGHT;
		public int angle = VoipImage.VIDEO_ROTATION_270;

		public CaptureParam(int width, int height) {
			this.width = width;
			this.height = height;
		}
	}

	interface VideoCaptureCallback {
		public void onVideoCaptureData(byte[] data, int width, int height,
									   boolean front, int angle, float faceLift);
	}

	private static final int MAX_SUPPORTED_WIDTH = 640;
	private static final int MAX_SUPPORTED_HEIGHT = 480;
	private SurfaceView previewView;
	private ViewGroup previewParentView;
	private Camera camera;
	private VideoCaptureCallback callback;
	private boolean isFront = true;
	private boolean hasSurfaceHolder;
	private CaptureParam param;
	private float facelift;
	private int targetFps = 20000;
	private int rotationOffset = 0;
	private boolean isPaused = false;
	private ReentrantLock cameraLock;
	private int previewWidth = 0;
	private int previewHeight = 0;
	private int capturedFrames = 0;
	private boolean previewStarted = false;

	/**
	 * Constructor.
	 * 
	 * @param callback
	 *            capture callback that should be VoipAndroid object
	 * @param param
	 *            camera capture parameter
	 */
	public VoipVideoCapture(VideoCaptureCallback callback, CaptureParam param) {
		this.callback = callback;
		if (null == param) {
			this.param = new CaptureParam(MAX_SUPPORTED_WIDTH,
					MAX_SUPPORTED_HEIGHT);
		} else {
			this.param = param;
		}
		this.cameraLock = new ReentrantLock();
	}

	/**
	 * Start video capturer. The camera will be start immediately if necessary,
	 * or will be started later automatically when needed.
	 * 
	 * @param frontCamera
	 *            if using front camera or back camera
	 * @param config
	 * 			   voip.getConfig           
	 *   
	 * @param parentView
	 *            the parent view group that the capture surface view will be
	 *            added on
	 * @return error code 0 : no error -1: camera has been started -2: camera is
	 *         disabled -3: open camera failed -4: cannot access camera
	 */
	public int start(boolean frontCamera,VoipConfig config, ViewGroup parentView) {
		cameraLock.lock();

		previewWidth = 0;
		previewHeight = 0;
		capturedFrames = 0;

		if (camera != null) {
			VoipLog.w("[CAMERA] Camera has been started! frontCamera="
					+ frontCamera + ", isFront=" + isFront);
			cameraLock.unlock();
			return -1;
		}

		DevicePolicyManager mDPM = (DevicePolicyManager) parentView
				.getContext().getSystemService(Context.DEVICE_POLICY_SERVICE);
		if (mDPM.getCameraDisabled(null)) {
			VoipLog.w("[CAMERA] Camera is disabled");
			cameraLock.unlock();
			return -2;
		}
		
		if(frontCamera) {
			param.angle = config.videoAngle;
		} else {
			param.angle = config.videoAngleBack;
		}
		

		isFront = frontCamera;
		int cameraid = isFront ? CameraInfo.CAMERA_FACING_FRONT
				: CameraInfo.CAMERA_FACING_BACK;

		try {
			camera = Camera.open(cameraid);
			if(camera != null) {
				Parameters camParam = camera.getParameters();

				// resolution setting
				Size captureSize = calculateSuitableSize();
				if (captureSize != null) {
					VoipLog.i("[CAMERA] capture resolution: " + captureSize.width + "x"
							+ captureSize.height);

					camParam.setPreviewSize(captureSize.width, captureSize.height);
				}

				// fps setting
				List<int[]> supportedFps = camParam.getSupportedPreviewFpsRange();
				if (supportedFps == null) {
					VoipLog.w("[CAMERA] getSupportedPreviewFpsRange() return null");
				} else {
					int[] selectedFps = new int[2];
					selectedFps[0] = selectedFps[1] = 0;

					for (int[] fpsRange : supportedFps) {
						if (fpsRange[1] >= targetFps) {
							if (selectedFps[1] < targetFps || selectedFps[0] <= fpsRange[0]) {
								selectedFps = fpsRange.clone();
								if (selectedFps[0] >= targetFps) {
									break;
								}
							}
						} else if (fpsRange[1] >= selectedFps[1]){
							selectedFps = fpsRange.clone();
						}
					}

					VoipLog.i("[CAMERA] capture fpsMin=" + selectedFps[0] + ", fpsMax=" + selectedFps[1]);
					camParam.setPreviewFpsRange(selectedFps[0], selectedFps[1]);
				}

				// auto-focus setting
				if (!isFront && !android.os.Build.MODEL.equals("M040")) {
					camParam.setFocusMode(Parameters.FOCUS_MODE_CONTINUOUS_VIDEO);
				}

				camera.setParameters(camParam);


			}

		} catch (RuntimeException e1) {
			e1.printStackTrace();
			try {
				//open camera failed, may be this android device has only one camera.
				camera = Camera.open();
				isFront = false;
				param.angle = config.videoAngleBack;
			} catch (RuntimeException e2) {
				e2.printStackTrace();
				cameraLock.unlock();
				return -3;
			}
		}

		if (null == camera) {
			VoipLog.e("[CAMERA] Cannot open camera!");
			cameraLock.unlock();
			return -4;
		}

		isPaused = false;
		previewView = new SurfaceView(parentView.getContext());
		previewView.getHolder().addCallback(this);
		previewParentView = parentView;
		try {
			previewParentView.addView(previewView, 0,
					new FrameLayout.LayoutParams(2, 2));
		} catch (Exception e) {
			VoipLog.d("[CAMERA] start camera fail " + e.getMessage());
		}

		VoipLog.i("[CAMERA] start camera " + cameraid);

		cameraLock.unlock();
		return 0;
	}

	/**
	 * Stop the camera immediately.
	 */
	public void stop() {
		cameraLock.lock();
		if (camera != null) {
			VoipLog.i("[CAMERA] stop camera");

			try {
				previewParentView.removeView(previewView);
				previewView.getHolder().removeCallback(this);
			} catch (Exception e) {
				VoipLog.e("[CAMERA][stop]" + e.getMessage());
			} finally {
				if (! previewStarted) {
					stopPreview();
				}
				previewParentView = null;
				previewView = null;
				camera.release();
				camera = null;
			}
		}

		previewWidth = 0;
		previewHeight = 0;

		cameraLock.unlock();
	}

	public void pause() {
		isPaused = true;
	}

	public void resume() {
		isPaused = false;
	}

	public void setFaceLift(float facelift) {
		this.facelift = facelift;
	}

	public void setRotationOffset(int offset) {
		this.rotationOffset = offset;
	}

	private void tryStartPreview() throws Exception {
		if (hasSurfaceHolder) {
			startPreview();
		}
	}

	private Size calculateSuitableSize() {

		List<Size> supportedSizes = camera.getParameters()
				.getSupportedPreviewSizes();

		if (supportedSizes == null) {
			VoipLog.w("[CAMERA] getSupportedPreviewSizes() return null");
			return null;
		}

		for (Size size : supportedSizes) {
			VoipLog.i("[CAMERA] camera supported size: " + size.width + "x"
					+ size.height);
		}

		int targetWidth = param.width;
		int targetHeight = param.height;
		Size captureSize = null;

		if (supportedSizes != null) {
			for (Size availableSize : supportedSizes) {
				if ((availableSize.width == targetWidth)
						&& (availableSize.height == targetHeight)) {
					captureSize = availableSize;
					break;
				}
			}

			if (captureSize == null) {
				for (Size availableSize : supportedSizes) {
					if ((availableSize.width == 2 * targetWidth)
							&& (availableSize.height == 2 * targetHeight)) {
						captureSize = availableSize;
						break;
					}
				}
			}

			if (captureSize == null) {
				for (Size availableSize : supportedSizes) {
					if ((availableSize.width == targetWidth)
							|| (availableSize.height == targetHeight)) {
						captureSize = availableSize;
						break;
					}
				}
			}

			if (captureSize == null) {
				int delta_size = Integer.MAX_VALUE;

				for (Size availableSize : supportedSizes) {
					if ((availableSize.width < targetWidth)
							&& (availableSize.height < targetHeight)
							&& (targetWidth * targetHeight
									- availableSize.width
									* availableSize.height < delta_size)) {
						captureSize = availableSize;
					}
				}
			}

			if (captureSize == null) {
				int delta_size = Integer.MAX_VALUE;

				for (Size availableSize : supportedSizes) {
					if ((availableSize.width >= targetWidth)
							&& (availableSize.height >= targetHeight)
							&& (availableSize.width * availableSize.height
									- targetWidth * targetHeight < delta_size)) {
						captureSize = availableSize;
					}
				}
			}

			if (captureSize == null) {
				int size = Integer.MAX_VALUE;

				for (Size availableSize : supportedSizes) {
					if (availableSize.width * availableSize.height < size) {
						captureSize = availableSize;
					}
				}
			}
		}

		if (captureSize == null) {
			VoipLog.w("[CAMERA] camera not support width/height:" + targetWidth
					+ "/" + targetHeight);
		} else if (captureSize.width != targetWidth || captureSize.height != targetHeight) {
			VoipLog.w("[CAMERA] camera not support width/height:" + targetWidth
					+ "/" + targetHeight + ", use " + captureSize.width + "/" + captureSize.height + " instead.");
		}

		return captureSize;
	}

	private void startPreview() throws Exception {
		VoipLog.i("[CAMERA] start camera preview");

		camera.setPreviewDisplay(previewView.getHolder());

		int width = MAX_SUPPORTED_WIDTH;
		int height = MAX_SUPPORTED_HEIGHT;
		Parameters camParam = camera.getParameters();
		if (camParam == null) {
			VoipLog.w("[CAMERA] getParameters() return null");
		} else {
			// resolution setting
			Size captureSize = calculateSuitableSize();
			if (captureSize != null) {
				width = captureSize.width;
				height = captureSize.height;
				VoipLog.i("[CAMERA] capture resolution: " + width + "x"
						+ height);

				camParam.setPreviewSize(width, height);
			}

			// fps setting
			List<int[]> supportedFps = camParam.getSupportedPreviewFpsRange();
			if (supportedFps == null) {
				VoipLog.w("[CAMERA] getSupportedPreviewFpsRange() return null");
			} else {
				int[] selectedFps = new int[2];
				selectedFps[0] = selectedFps[1] = 0;

				for (int[] fpsRange : supportedFps) {
					if (fpsRange[1] >= targetFps) {
						if (selectedFps[1] < targetFps || selectedFps[0] <= fpsRange[0]) {
							selectedFps = fpsRange.clone();
							if (selectedFps[0] >= targetFps) {
								break;
							}
						}
					} else if (fpsRange[1] >= selectedFps[1]){
						selectedFps = fpsRange.clone();
					}
				}

				VoipLog.i("[CAMERA] capture fpsMin=" + selectedFps[0] + ", fpsMax=" + selectedFps[1]);
				camParam.setPreviewFpsRange(selectedFps[0], selectedFps[1]);
			}

			// auto-focus setting
			if (!isFront && !android.os.Build.MODEL.equals("M040")) {
				camParam.setFocusMode(Parameters.FOCUS_MODE_CONTINUOUS_VIDEO);
			}

			try {
				camera.setParameters(camParam);
			} catch (Exception e) {
				VoipLog.e("setParameters " + e.getMessage());
			}
		}

		byte[][] buffer = new byte[3][width * height * 3 / 2];
		for (byte[] item : buffer) {
			camera.addCallbackBuffer(item);
		}

		try {
			camera.setPreviewCallbackWithBuffer(this);
		} catch (Exception e) {

		}

		try {
			camera.startPreview();
		} catch (Exception e) {
			VoipLog.e("startPreview " + e.getMessage());
		}
		previewStarted = true;
	}

	public void updateParams(CaptureParam param) {
		this.param = param;
	}

	private void stopPreview() {
		VoipLog.i("[CAMERA] stop camera preview");

		cameraLock.lock();
		if (camera != null) {
			try {
				camera.stopPreview();
			} catch (Exception e) {
				VoipLog.i("[CAMERA] stop camera preview " + e.getMessage());
			}
			try {
				camera.setPreviewCallbackWithBuffer(null);
				camera.setPreviewCallback(null);
				camera.setPreviewDisplay(null);
			} catch (Exception e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}
		}
		previewStarted = false;
		cameraLock.unlock();
	}

	@Override
	public void onPreviewFrame(byte[] data, Camera camera) {
		try {
			if (previewWidth == 0 || previewHeight == 0) {
				Size size = camera.getParameters().getPreviewSize();
				previewWidth = size.width;
				previewHeight = size.height;
				VoipLog.d("[CAMERA] onPreviewFrame, width=" + previewWidth + ", height=" + previewHeight);
			}

			if (previewWidth * previewHeight * 3 / 2 != data.length) {
				camera.addCallbackBuffer(data);
				return;
			}

			if (callback != null && false == isPaused) {
				callback.onVideoCaptureData(data, previewWidth, previewHeight, isFront,
						(param.angle + this.rotationOffset) % 4, facelift);
			}

			if (capturedFrames++ % 150 == 0) {
				VoipLog.d("[CAMERA] onPreviewFrame, width=" + previewWidth + ", height=" + previewHeight
						+ ", isPaused=" + isPaused + ", capturedFrames=" + capturedFrames);
			}

			camera.addCallbackBuffer(data);
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	@Override
	public void surfaceChanged(SurfaceHolder holder, int format, int width,
			int height) {
		// Empty
	}

	@Override
	public void surfaceCreated(SurfaceHolder holder) {
		cameraLock.lock();
		VoipLog.i("[CAMERA] preview surface created");
		previewWidth = 0;
		previewHeight = 0;
		hasSurfaceHolder = true;

		try {
			tryStartPreview();
		} catch (Exception e) {
			e.printStackTrace();
			VoipLog.e("[CAMERA] start preview error");
			hasSurfaceHolder = false;
		}
		cameraLock.unlock();
	}

	@Override
	public void surfaceDestroyed(SurfaceHolder holder) {
		cameraLock.lock();
		VoipLog.i("[CAMERA] preview surface destroyed");
		hasSurfaceHolder = false;

		if (camera != null) {
			stopPreview();
		}

		previewWidth = 0;
		previewHeight = 0;
		cameraLock.unlock();
	}
}
