package me.chatgame.voip;

import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.microedition.khronos.egl.EGL10;
import javax.microedition.khronos.egl.EGLConfig;
import javax.microedition.khronos.egl.EGLContext;
import javax.microedition.khronos.egl.EGLDisplay;
import javax.microedition.khronos.opengles.GL10;

import android.content.Context;
import android.opengl.GLSurfaceView;

public class VoipVideoRender extends GLSurfaceView implements
		GLSurfaceView.Renderer {

	public static final int VIDEO_ROTATION_0 = 0;
	public static final int VIDEO_ROTATION_90 = 1;
	public static final int VIDEO_ROTATION_180 = 2;
	public static final int VIDEO_ROTATION_270 = 3;

	private long render;
	private boolean mirror;
	private VoipImage lastFrame;
	private List<VoipImage> jitterBufs;
	private Lock jitterBufsLock;
	private static final int JITTER_MAX = 10;
	private static final int RENDER_FPS = 20;
	private long nextRenderTime = 0;
	private long renderGap = 0;
	private String objectName = "";

	public VoipVideoRender(Context context, boolean isMirror, String name) {
		super(context);
		render = createRender();
		mirror = isMirror;
		lastFrame = null;
		jitterBufsLock = new ReentrantLock();
		jitterBufs = new LinkedList<VoipImage>();
		nextRenderTime = 0;
		renderGap = 0;
		objectName = name;
		init();
	}

	public void setMirror(boolean enable) {
		mirror = enable;
	}

	public void pushVideoFrame(VoipImage image) {
		jitterBufsLock.lock();

		jitterBufs.add(image);
		if (jitterBufs.size() > JITTER_MAX) {
			VoipLog.w("pushVideoFrame: too many frames in jitterBufs of " + objectName + ", drop some of them");
			for (int i = 0; i < JITTER_MAX / 2; ++i) {
				jitterBufs.remove(0);
			}
		}

		jitterBufsLock.unlock();
	}

	private void init() {
		setEGLContextFactory(new ContextFactory());
		setEGLContextClientVersion(2);
		setRenderer(this);
	}

	protected void finalize() throws Throwable {
		super.finalize();
		destroyRender(render);
	}

	private static class ContextFactory implements
			EGLContextFactory {
		private static int EGL_CONTEXT_CLIENT_VERSION = 0x3098;

		public EGLContext createContext(EGL10 egl, EGLDisplay display,
				EGLConfig eglConfig) {
			int[] attrib_list = { EGL_CONTEXT_CLIENT_VERSION, 2, EGL10.EGL_NONE };
			EGLContext context = egl.eglCreateContext(display, eglConfig,
					EGL10.EGL_NO_CONTEXT, attrib_list);
			return context;
		}

		public void destroyContext(EGL10 egl, EGLDisplay display,
				EGLContext context) {
			egl.eglDestroyContext(display, context);
		}
	}

	@Override
	public void onDrawFrame(GL10 gl) {
		long currTime = System.currentTimeMillis();

		if (nextRenderTime == 0) {
			nextRenderTime = currTime + 1000 / RENDER_FPS;
		} else {
			if (nextRenderTime > currTime) {
				try {
					Thread.sleep(nextRenderTime - currTime);
				} catch (InterruptedException e) {
					// TODO Auto-generated catch block
					e.printStackTrace();
				}
			}
			nextRenderTime += 1000 / RENDER_FPS;
		}
		
		VoipImage image = null;

		jitterBufsLock.lock();
		if (!jitterBufs.isEmpty()) {
			image = jitterBufs.get(0);
			jitterBufs.remove(0);
		}
		jitterBufsLock.unlock();

		if (image == null) {
			image = lastFrame;
		}

		if (image != null) {
			renderDraw(render, image.data, image.width, image.height,
					image.rotation, mirror, image.facelift);
			lastFrame = image;
		}
	}

	@Override
	public void onSurfaceChanged(GL10 gl, int width, int height) {
		renderResize(render, width, height);
	}

	@Override
	public void onSurfaceCreated(GL10 gl, EGLConfig config) {
		init(render);
	}

	private native long createRender();

	private native void init(long render);

	private native void destroyRender(long render);

	private native void renderDraw(long render, byte[] data, int width,
			int height, int rotation, boolean mirror, float facelift);

	private native void renderResize(long render, int width, int height);
}
