/*
 * Copyright 2014 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package me.cmoz.gradle.snapshot;

import lombok.SneakyThrows;
import org.gradle.api.DefaultTask;
import org.gradle.api.plugins.ExtensionContainer;
import org.gradle.api.tasks.TaskAction;

import javax.annotation.Nullable;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Map;
import java.util.Properties;

import static java.util.Map.Entry;

/**
 * A task to extract build information from an SCM repository and generate a
 * file with the most recent commit data.
 */
public class SnapshotTask extends DefaultTask {

    /** An output label for the properties file. */
    private static final String BUILD_LABEL =
            "Generated by the Gradle Snapshot Plugin ("
            + SnapshotTask.class.getPackage().getImplementationVersion()
            + ")";

    /**
     * The main action for this task.
     *
     * <p>The task generates metadata from the SCM for the most recent commit
     * and stores it to the {@code snapshot} file location for storing with
     * JAR and WAR packages.
     */
    @TaskAction
    @SneakyThrows(IOException.class)
    public void action() {
        final ExtensionContainer extensions = getProject().getExtensions();
        final SnapshotPluginExtension ext = extensions.getByType(SnapshotPluginExtension.class);

        final SCMCommand scmCmd = getSCMCommand();
        if (scmCmd == null) {
            System.out.println("  No supported SCM repository found, skipping task.");
            return;
        }

        final Properties properties = new Properties();
        final Map<String, String> commitMap = scmCmd.getLatestCommit(ext.getDateFormat()).asMap();

        for (final Entry<String, String> entry : commitMap.entrySet()) {
            // make the build information available to the project properties
            extensions.getExtraProperties().set(entry.getKey(), entry.getValue());
            if (ext.isVerbose()) {
                System.out.println("  " + entry);
            }
        }
        properties.putAll(commitMap);

        final File outputDir = new File(getProject().getBuildDir(), "snapshot");
        outputDir.mkdirs();
        final File output = new File(outputDir, ext.getFilename());
        properties.store(new FileWriter(output), BUILD_LABEL);
    }

    /**
     * Searches the project directory for a repository directory that belongs
     * to a supported SCM tool. If one is found the associated {@code SCMCommand}
     * will be returned to interact with it.
     *
     * @return The {@code SCMCommand} to interact with the SCM repository for
     *         the project, if one could be found.
     */
    @Nullable
    private SCMCommand getSCMCommand() {
        SCMCommand scmCmd = new HgSCMCommand(getProject());
        if (scmCmd.getRepositoryDir() != null) {
            return scmCmd;
        }

        scmCmd = new GitSCMCommand(getProject());
        if (scmCmd.getRepositoryDir() != null) {
            return scmCmd;
        }

        return null; // no supported SCM directory could be located
    }

}
