package me.ehp246.aufjms.core.reflection;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Stream;

import me.ehp246.aufjms.api.reflection.Invocation;

/**
 * 
 * @author Lei Yang
 * @since 1.0
 */
public final class DefaultProxyInvocation implements Invocation {
    private final Class<?> declaringType;
    private final Object target;
    private final Method method;
    private final List<?> args;
    private final Annotation[][] parameterAnnotations;
    private final List<Class<?>> threws;
    private final Parameter[] parameters;

    public DefaultProxyInvocation(final Class<?> declaringType, final Object target, final Method method,
            final Object[] args) {
        this.declaringType = declaringType;
        this.target = target;
        this.method = Objects.requireNonNull(method);
        this.args = Collections.unmodifiableList(args == null ? List.of() : Arrays.asList(args));
        this.parameterAnnotations = this.method.getParameterAnnotations();
        this.threws = List.of(this.method.getExceptionTypes());
        this.parameters = method.getParameters();
    }

    public Class<?> declaringType() {
        return this.declaringType;
    }

    @Override
    public Object target() {
        return target;
    }

    @Override
    public Method method() {
        return method;
    }

    public List<? extends Annotation> getMethodDeclaredAnnotations() {
        return List.of(method.getDeclaredAnnotations());
    }

    public Class<?> getDeclaringClass() {
        return method.getDeclaringClass();
    }

    public String getDeclaringClassSimpleName() {
        return method.getDeclaringClass().getSimpleName();
    }

    @Override
    public List<?> args() {
        return args;
    }

    public Class<?> getReturnType() {
        return this.method.getReturnType();
    }

    public boolean hasReturn() {
        return this.method.getReturnType() != void.class && this.method.getReturnType() != Void.class;
    }

    public boolean isAsync() {
        return getReturnType().isAssignableFrom(CompletableFuture.class);
    }

    public boolean isSync() {
        return !isAsync();
    }

    /**
     * Void is considered a declared return.
     *
     * @return
     */
    public List<Class<?>> getThrows() {
        return threws;
    }

    public boolean canThrow(Class<?> type) {
        return this.getThrows().stream().filter(t -> t.isAssignableFrom(type)).findAny().isPresent();
    }

    public boolean canReturn(Class<?> type) {
        return this.method.getReturnType().isAssignableFrom(type);
    }

    public List<ReflectedArgument> filterPayloadArgs(
            final Set<Class<? extends Annotation>> exclusions) {
        final var valueArgs = new ArrayList<ReflectedArgument>();
        for (var i = 0; i < parameterAnnotations.length; i++) {
            if (Stream.of(parameterAnnotations[i])
                    .filter(annotation -> exclusions.contains(annotation.annotationType())).findAny().isPresent()) {
                continue;
            }
            valueArgs.add(new ReflectedArgument(args.get(i), this.parameters[i], method));
        }

        return valueArgs;
    }

    /**
     * Returns the value of the annotation or default if annotation is not found.
     */
    public <A extends Annotation, V> Optional<A> findOnDeclaringClass(final Class<A> annotationClass) {
        return Optional.ofNullable(this.method.getDeclaringClass().getAnnotation(annotationClass));
    }

    /**
     * Find all arguments of the given parameter type.
     *
     * @param <R>  Parameter type
     * @param type Class of the parameter type
     * @return all arguments of the given type. Could have <code>null</code>.
     */
    @SuppressWarnings("unchecked")
    public <R> List<R> findArgumentsOfType(final Class<R> type) {
        final var list = new ArrayList<R>();
        final var parameterTypes = method.getParameterTypes();
        for (int i = 0; i < parameterTypes.length; i++) {
            if (type.isAssignableFrom(parameterTypes[i])) {
                list.add((R) args.get(i));
            }
        }
        return list;
    }

    /**
     * Looks for arguments that are annotated by the given Annotation type. Returns
     * a map with the key provided by the key supplier function, the value the
     * argument.
     *
     * @param <K>            Key from the key supplier
     * @param <V>            Argument object reference
     * @param <A>            Annotation type
     * @param annotationType
     * @param keySupplier
     * @return returned Map can be modified. Never <code>null</code>.
     */
    @SuppressWarnings("unchecked")
    public <K, V, A extends Annotation> Map<K, V> mapAnnotatedArguments(final Class<A> annotationType,
            final Function<A, K> keySupplier) {
        final var map = new HashMap<K, V>();
        for (int i = 0; i < parameterAnnotations.length; i++) {
            final var found = Stream.of(parameterAnnotations[i])
                    .filter(annotation -> annotation.annotationType() == annotationType).findFirst();
            if (found.isEmpty()) {
                continue;
            }

            map.put(keySupplier.apply((A) found.get()), (V) args.get(i));
        }
        return map;
    }

    @SuppressWarnings("unchecked")
    public <A extends Annotation> Stream<AnnotatedArgument<A>> streamOfAnnotatedArguments(
            final Class<A> annotationType) {
        final var builder = Stream.<AnnotatedArgument<A>>builder();

        for (int i = 0; i < parameterAnnotations.length; i++) {
            final var arg = args.get(i);
            final var parameter = method.getParameters()[i];
            Stream.of(parameterAnnotations[i]).filter(annotation -> annotation.annotationType() == annotationType)
                    .map(anno -> new AnnotatedArgument<A>((A) anno, arg, parameter)).forEach(builder::add);
            ;
        }

        return builder.build();
    }

    public <A extends Annotation, V> Optional<V> optionalValueOnMethod(final Class<A> annotationClass,
            final Function<A, V> mapper) {
        return Optional.ofNullable(this.findOnMethod(annotationClass).map(mapper).orElse(null));
    }

    public <A extends Annotation, V> V firstArgumentAnnotationOf(final Class<A> annotationClass,
            final Function<AnnotatedArgument<A>, V> mapper, final Supplier<V> supplier) {
        final var found = this.streamOfAnnotatedArguments(annotationClass).findFirst();
        return found.isPresent() ? mapper.apply(found.get()) : supplier.get();
    }

    public <A extends Annotation, V> V methodAnnotationOf(final Class<A> annotationClass, final Function<A, V> mapper,
            final Supplier<V> supplier) {
        final var found = this.findOnMethod(annotationClass);
        return found.isPresent() ? mapper.apply(found.get()) : supplier.get();
    }

    public <A extends Annotation, V> V classAnnotationOf(final Class<A> annotationClass, final Function<A, V> mapper,
            final Supplier<V> supplier) {
        final var found = this.findOnDeclaringClass(annotationClass);
        return found.isPresent() ? mapper.apply(found.get()) : supplier.get();
    }

    /**
     * Resolve the annotation value up the invocation hierarchy.
     */
    public <A extends Annotation, V> V resolveAnnotatedValue(final Class<A> annotationClass,
            final Function<AnnotatedArgument<A>, V> argMapper, final Function<A, V> methodMapper,
            final Function<A, V> classMapper, final Supplier<V> supplier) {
        return firstArgumentAnnotationOf(annotationClass, argMapper, () -> methodAnnotationOf(annotationClass,
                methodMapper, () -> classAnnotationOf(annotationClass, classMapper, supplier)));
    }

    public <A extends Annotation> Optional<A> findOnMethod(final Class<A> annotationClass) {
        return Optional.ofNullable(method.getAnnotation(annotationClass));
    }

    public <A extends Annotation> Optional<A> findOnMethodUp(final Class<A> annotationClass) {
        final var found = Optional.ofNullable(method.getAnnotation(annotationClass));
        if (found.isPresent()) {
            return found;
        }

        return Optional.ofNullable(getDeclaringClass().getAnnotation(annotationClass));
    }

    public String getMethodName() {
        return method.getName();
    }
}
