/*
 * Copyright (C) 2020 AriaLyy(https://github.com/AriaLyy/KeepassA)
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, you can obtain one at http://mozilla.org/MPL/2.0/.
 */
package com.lyy.keepassa.widget

import android.animation.Animator
import android.animation.ValueAnimator
import android.content.Context
import android.graphics.*
import android.util.AttributeSet
import android.view.View
import android.view.animation.Animation
import android.view.animation.LinearInterpolator
import androidx.annotation.FloatRange
import kotlin.math.ceil

open class PacManView(context: Context, attrs: AttributeSet? = null) : View(context, attrs),
    ValueAnimator.AnimatorUpdateListener, Animator.AnimatorListener {

    private var DEFAULT_SIZE = 56.0f
    private val ANIMATION_START_DELAY: Long = 333
    private val ANIMATION_DURATION: Long = 1333

    private var mAllSize = 0f
    private var mViewWidth = 0f
    private var mViewHeight = 0f

    private lateinit var mFloatValueAnimator: ValueAnimator
    private var mDurationTimePercent = 1.0

    //最终阶段
    private val FINAL_STATE = 9
    private val MAX_MOUTH_ANGLE = 45
    private var mDurationTime: Long = 333
    private lateinit var mFullPaint: Paint
    private lateinit var mOuterCircleRectF: RectF
    private var mMouthAngle = 0
    private var mMoveDistance = 0f

    //当前动画阶段
    private var mCurrAnimatorState = 0
    private var HorizontalAngle = 0
    private var mMaxMoveRange = 0f
    private var mLastMoveDistance = 0f
    private var mDefaultStartMoveX = 0f

    override fun onFinishInflate() {
        super.onFinishInflate()
        initAnimators()

        val outR: Float = mAllSize
        val inR = outR * 0.7f
        mMaxMoveRange = mViewWidth + 2 * inR //移动距离范围

        initPaint() //圆范围

        mMouthAngle = MAX_MOUTH_ANGLE //嘴度数
        HorizontalAngle = 0 //水平翻转度数
        mDefaultStartMoveX = -mMaxMoveRange * 0.5f //默认偏移量
        mMoveDistance = 0f //移动距离
        mOuterCircleRectF = RectF(
            getViewCenterX() - inR,
            getViewCenterY() - inR,
            getViewCenterX() + inR,
            getViewCenterY() + inR
        )
    }

    private fun initAnimators() {
        mAllSize = (DEFAULT_SIZE * 0.5f - 12).toPx().toFloat()
        mViewWidth = DEFAULT_SIZE.toPx().toFloat()
        mViewHeight = DEFAULT_SIZE.toPx().toFloat()
        mFloatValueAnimator = ValueAnimator.ofFloat(0.0f, 1.0f)
        mFloatValueAnimator.repeatCount = Animation.INFINITE
        mFloatValueAnimator.duration = getAnimationDuration()
        mFloatValueAnimator.startDelay = ANIMATION_START_DELAY
        mFloatValueAnimator.interpolator = LinearInterpolator()
    }

    private fun initPaint() {
        mFullPaint = Paint(Paint.ANTI_ALIAS_FLAG)
        mFullPaint.style = Paint.Style.FILL
        mFullPaint.color = Color.WHITE
        mFullPaint.isDither = true
        mFullPaint.isFilterBitmap = true
        mFullPaint.strokeCap = Paint.Cap.ROUND
        mFullPaint.strokeJoin = Paint.Join.ROUND
    }


    override fun onDraw(canvas: Canvas) {
        canvas.save()
        canvas.translate(mDefaultStartMoveX + mMoveDistance, 0f)
        canvas.rotate(HorizontalAngle.toFloat(), getViewCenterX(), getViewCenterY())
        canvas.drawArc(
            mOuterCircleRectF,
            mMouthAngle.toFloat(),
            (360 - mMouthAngle * 2).toFloat(),
            true,
            mFullPaint
        )
        canvas.restore()
    }

    private fun setAlpha(alpha: Int) {
        mFullPaint.alpha = alpha
    }

    private fun prepareStart(floatValueAnimator: ValueAnimator) {
        mDurationTime = ceil(getAnimationDuration() * 0.3).toLong()
        floatValueAnimator.duration = mDurationTime
    }

    private fun computeUpdateValue(
        animation: ValueAnimator,
        @FloatRange(from = 0.0, to = 1.0) animatedValue: Float
    ) {
        val half = FINAL_STATE / 2 + 1
        val step = mMaxMoveRange / half
        if (mCurrAnimatorState < half) //以下分为两个阶段
        { //向右
            HorizontalAngle = 0
            mMoveDistance = mLastMoveDistance + step * animatedValue
        } else { //向左
            HorizontalAngle = 180
            mMoveDistance = mLastMoveDistance - step * animatedValue
        }
        //嘴张开度数
        mMouthAngle = if (mCurrAnimatorState % 2 == 0) {
            (MAX_MOUTH_ANGLE * animatedValue).toInt() + 5
        } else {
            (MAX_MOUTH_ANGLE * (1 - animatedValue)).toInt() + 5
        }
    }

    fun start() {
        if (mFloatValueAnimator.isStarted) {
            return
        }
        mFloatValueAnimator.addUpdateListener(this)
        mFloatValueAnimator.addListener(this)
        mFloatValueAnimator.repeatCount = Animation.INFINITE
        mFloatValueAnimator.duration = getAnimationDuration()
        prepareStart(mFloatValueAnimator)
        mFloatValueAnimator.start()
    }

    fun stop() {
        mFloatValueAnimator.removeAllUpdateListeners()
        mFloatValueAnimator.removeAllListeners()
        mFloatValueAnimator.repeatCount = 0
        mFloatValueAnimator.duration = 0
        mFloatValueAnimator.end()
    }

    fun isRunning(): Boolean {
        return mFloatValueAnimator.isRunning
    }

    override fun onAnimationUpdate(animation: ValueAnimator) {
        computeUpdateValue(animation, animation.animatedValue as Float)
    }


    override fun onAnimationStart(animation: Animator) {
    }

    override fun onAnimationEnd(animation: Animator) {
    }

    override fun onAnimationCancel(animation: Animator) {
    }

    override fun onAnimationRepeat(animation: Animator) {
        if (++mCurrAnimatorState > FINAL_STATE) { //还原到第一阶段
            mCurrAnimatorState = 0
        }
        //矫正
        val half = FINAL_STATE / 2 + 1
        val stepRange = mMaxMoveRange / half
        mLastMoveDistance = if (mCurrAnimatorState < half) {
            stepRange * mCurrAnimatorState
        } else {
            stepRange * (half - mCurrAnimatorState % half)
        }
    }

    private fun setColorFilter(colorFilter: ColorFilter) {
        mFullPaint.colorFilter = colorFilter
    }

    fun setDurationTimePercent(durationTimePercent: Double) {
        mDurationTimePercent = if (durationTimePercent <= 0) {
            1.0
        } else {
            durationTimePercent
        }
    }


    private fun getAnimationDuration(): Long {
        return ceil(ANIMATION_DURATION * mDurationTimePercent).toLong()
    }


    private fun getViewCenterX(): Float {
        return mViewWidth * 0.5f
    }

    private fun getViewCenterY(): Float {
        return mViewHeight * 0.5f
    }

}